/*
 * $Id: ARMI.java,v 1.2 2003/09/24 14:55:50 cactushack76 Exp $
 */

package com.navtools.armi;

/**
 * Asynchronous RMI Subsystem Interface
 *
 * Unless you are extending the ARMI system, the static methods in this class
 * should supply you everything you need to use the ARMI system.  The ARMI
 * system is a replacement for Java RMI.  The advantages of ARMI are:
 * 1) Network messages are _much_ more compact.
 * 2) Return values can be received synchronously or asynchronously.
 * 3) UDP networking is used, so messages are sent faster (less low level handshaking).
 * 4) Since UDP networking is used, ARMI can be used over multicast.  This is
 *    currently untested.
 *
 * the disadvantages as compared to RMI are:
 * 1) ARMI is a little harder to use:
 *     a) all methods in shared interfaces must take as a first parameter
 *        a long msgID, which must be unique for the client session.  Just
 *        use ARMI.getNextMessageID() to retrieve compliant IDs.
 *     b) objects to be sent over the wire as copies (equivalent to sending
 *        Serializable objects over the wire in RMI) must implement
 *        DataStreamable.  They must implement a method to do a minimal
 *        'Serialization' of their data to and from a DataStream.
 *     c) The values returned by ARMI methods do not come back immediately;
 *        you must either wait for the return value or register code to
 *        handle the return when it comes back.
 * 2) Currently, lost messages are just thrown away.  Obviously this is
 *    typically unacceptable, and I'm working on a UDP wrapper class to
 *    ensure that messages are received in order and lost messages are re-sent.
 *    This will be optional, but will be the default.  You will still be able
 *    to choose to have slightly faster networking that is unsafe.
 *
 * Note: Temporarily, ARMI is using TCP instead of UDP.  So the advantages of
 * using UDP are obviously not gained by ARMI.  However, point 2 about
 * messages being thrown away is also not true.  -Bobby
 */

import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Set;

import com.navtools.armi.networking.Message;
import com.navtools.armi.networking.ServerAddress;
import com.navtools.armi.networking.UDPMessengerInterface;
import org.apache.log4j.Category;

//import java.rmi.UnknownHostException;

public class ARMI
{
	public static final Category LOG =
	Category.getInstance( ARMI.class.getName() );


	/**
	 * Call this method once, typically in main, to establish the program
	 * as an ARMI client program.  Use this for all ARMI progs which won't be
	 * running the registry server.
	 */
	public static void initialize( InetAddress registryAddress )
	throws SocketException, UnknownHostException, Exception
	{
		ARMIProxyFactory.initialize( registryAddress );
	}


	public static UDPMessengerInterface getMessenger()
	{
		return ARMIProxyFactory.getMessenger();
	}


	/**
	 * Call this method once, typically in main, to establish the
	 * ARMI registry server.
	 */
	public static void initializeAsRegistryServer()
	throws SocketException, UnknownHostException, Exception
	{
		ARMIProxyFactory.initializeAsRegistryServer();
	}


	/**
	 * Start listening for remote method calls on obj.  You must call this
	 * method (normally via ARMI.share()) exactly once on each object you
	 * want to be accessible remotely.
	 *
	 * This is just like the Naming.rebind that you would do in RMI to share
	 * a UnicastRemoteObject.
	 *
	 * @param obj the object which will receive remote method calls
	 * @param remoteInterfaceForObj the interface defining the calls on obj
	 * that can be made remotely
	 * @return an integer indicating the ID that obj will have on this ARMI
	 * server.  It is the number of objects implementing interfaceImplemented
	 * that have been shared so far on the server.  Any client who knows
	 * the IP and port of the server, the interfaceImplemented, and the ID
	 * can call methods on obj remotely.
	 */
	public static int share( Object obj, Class remoteInterfaceForObj )
	{
		return RMIMessageListener.instance().
		addInstance( obj, remoteInterfaceForObj );
	}


	public static boolean isShared( Object obj )
	{
		return ARMIProxyFactory.isShared( obj );
	}


	public static Object waitForReturn( long messageID, Class returnType )
	{
		return RMIResponseMessageListener.instance().waitForReturn( messageID,
		                                                            returnType );
	}


	public static void waitForReturn( long messageID )
	{
		RMIResponseMessageListener.instance().waitForReturn( messageID );
	}


	/**
	 * Use this method to get a proxy for the singleton instance of the
	 * interface given on the server at the address given.  You will
	 * have to cast this returned object to be the type of intrface.
	 */
	static public Object getProxy( Class intrface, InetAddress inetAddress )
	{
		return ARMIProxyFactory.newInstance( intrface, inetAddress );
	}


	/**
	 * Use this method to get a proxy for the instance of the
	 * interface given on the server at the address given.  You will
	 * have to cast this returned object to be the type of intrface.
	 */
	static public Object getProxy( Class intrface, InetAddress inetAddress,
	                               int instanceID ) 
	{
		return ARMIProxyFactory.newInstance( intrface, inetAddress,
		                                     instanceID );
	}


	/**
	 * Use this method to get a proxy for the singleton instance of the
	 * interface given on the server at the address given.  You will
	 * have to cast this returned object to be the type of intrface.
	 */
	static public Object getProxy( Class intrface, ServerAddress address )
	{
		return ARMIProxyFactory.newInstance( intrface, address );
	}


	/**
	 * Use this method to get a proxy for the singleton instance of the
	 * interface given on the server at the address given.  You will
	 * have to cast this returned object to be the type of intrface.
	 */
	static public Object getProxy( Class intrface, ServerAddress address,
	                               int instanceID )
	{
		return ARMIProxyFactory.newInstance( intrface, address, instanceID );
	}


	static public long getNextMessageID()
	{
		return Message.getNextID();
	}


	static public Set getClients()
	{
		return ARMIProxyFactory.imq_.getClients();
	}


	/**
	 * Dummy implementation for now; just return a metadata that doesn't
	 * require response for methods that return void, except "init"
	 * TODO FIX BIG FAT KLUDGE
	 */
	static public MethodMetaData getMethodMetaData( Method method )
	{
		return method.getName().equals( "init" ) || !method.getReturnType().equals( void.class ) ?
		       MethodMetaData.REQUIRES_RESPONSE :
		       MethodMetaData.REQUIRES_NO_RESPONSE;
	}


	static abstract class MethodMetaData
	{
		abstract boolean requiresResponse();


		static public MethodMetaData REQUIRES_RESPONSE = new MethodMetaData()
		{
			public boolean requiresResponse()
			{
				return true;
			}
		};

		static public MethodMetaData REQUIRES_NO_RESPONSE = new MethodMetaData()
		{
			public boolean requiresResponse()
			{
				return false;
			}
		};
	}

	//debugging code follows
}

