/*
 * $Id: ClassAndMethodTable.java,v 1.2 2003/09/24 14:55:50 cactushack76 Exp $
 */

package com.navtools.armi;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.navtools.armi.networking.ServerAddress;
import com.navtools.thread.Lock;
import com.navtools.util.A;
import com.navtools.util.MethodMatcher;
import org.apache.log4j.Category;

public class ClassAndMethodTable
{
	public static final Category LOG =
	Category.getInstance( ClassAndMethodTable.class.getName() );

	protected ClassAndMethodTable( ServerAddress registryAddress )
	{
		registryAddress_ = registryAddress;

		//Registry must always get the same IDs so the ClassAndMethodTable
		//can negotiate the other IDs with the registry
		addMapping( new Integer( nextID_++ ), Registry.class, false );

		//must sort methods somehow so the ids are the same on all machines
		Method[] registryMethods = Registry.class.getMethods();
		String[] methodNames = new String[registryMethods.length];
		Map nameToMethod = new HashMap();
		for ( int i = 0; i < registryMethods.length; ++i )
		{
			String methodName = MethodMatcher.instance().
			conditionMethodName( registryMethods[i].toString() );

			nameToMethod.put( methodName, registryMethods[i] );
			methodNames[i] = methodName;
		}
		Arrays.sort( methodNames );

		//now register the sorted methods
		for ( int i = 0; i < methodNames.length; ++i )
		{
			addMapping( new Integer( nextID_++ ),
			            nameToMethod.get( methodNames[i] ), false );
		}

		//remember the max method id for the ClassAndMethodTable shared methods
		maxInfrastructureMethodID_ = nextID_ - 1;

		// new to prevent deadlock when getting return type for registry methods
		addMapping( new Integer( nextID_++ ), Integer.class, false );
		addMapping( new Integer( nextID_++ ), String.class, false );

		loadMappings();

		if ( registryAddress_ == null )
		{
			if ( !registryShared_ )
			{
				registryShared_ = true;

				RMIMessageListener.instance().
				addInstance( new ServerRegistry(), Registry.class );
				if ( LOG.isDebugEnabled() )
				{
					LOG.debug( "Shared ServerRegistry" );
				}
			}
		}
	}

	public static ClassAndMethodTable instance()
	{
		if ( instance_ == null )
		{
			Lock.logBeforeSync();
			synchronized ( ClassAndMethodTable.class )
			{
				Lock.logBeginSync();
				if ( instance_ == null )
				{
					instance_ = new ClassAndMethodTable( null );
				}
				Lock.logEndSync();
			}
			Lock.logAfterSync();
		}

		return instance_;
	}

	/**
	 * Run this to establish the ClassAndMethodTable before you use it for
	 * the first time, unless this is for the registryAddress machine.
	 * The registry server machine can either just start calling instance()
	 * or can call this method with null as a parameter.
	 */
	public static ClassAndMethodTable
	establishInstance( ServerAddress registryAddress )
	{
                // There may be a case where a client is connecting to a different server
                if (instance_ != null && registryAddress.equals(instance_.registryAddress_)) {
		    A.ssert( instance_ == null, "Instance already established" );
                }
		instance_ = new ClassAndMethodTable( registryAddress );
		return instance_;
	}

	public boolean isInfrastructureMethod( int methodID )
	{
		return methodID <= maxInfrastructureMethodID_;
	}

	public Integer getID( Class theClass )
	{
		return getID( (Object) theClass );
	}

	public Integer getID( Method method )
	{
		return getID( (Object) method );
	}

	protected Integer getID( Object classOrMethod )
	{
		Integer retval = (Integer) objectToIdMap_.get( classOrMethod );
		if ( retval == null )
		{
			getPairFromRegistryServer( null, classOrMethod );
			retval = (Integer) objectToIdMap_.get( classOrMethod );
		}
		return retval;
	}

	public String getName( Integer id )
	{
		String retval = getClassOrMethodName( getClassOrMethod( id ) );
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Returning name " + retval + " for " + id );
		}
		return retval;
	}

	protected static String getClassOrMethodName( Object classOrMethod )
	{
		String retval = null;

		if ( classOrMethod instanceof Class )
		{
			retval = ( (Class) classOrMethod ).getName();
		}
		else
		{
			if ( classOrMethod instanceof Method )
			{
				retval = MethodMatcher.instance().conditionMethodName( ( (Method) classOrMethod ).toString() );
			}
			else
			{
				LOG.error( "Error: classOrMethod is neither " +
				           "a Class or a Method, it is a " +
				           classOrMethod.getClass().getName() );
			}
		}

		return retval;
	}

	public Object getClassOrMethod( Integer id )
	{
		Object retval = idToObjectMap_.get( id );
		if ( retval == null )
		{
			getPairFromRegistryServer( id, null );
			retval = idToObjectMap_.get( id );
		}
		return retval;
	}

	public Class getClass( Integer id )
	{
		try
		{
			return (Class) getClassOrMethod( id );
		}
		catch ( RuntimeException e )
		{
			LOG.error( "Exception getting class for id " + id );
			throw e;
		}
	}

	public Method getMethod( Integer id )
	{
		try
		{
			return (Method) getClassOrMethod( id );
		}
		catch ( RuntimeException e )
		{
			LOG.error( "Exception getting method for id " + id );
			throw e;
		}
	}

	protected static Map requestsInQueue_ = new HashMap();

	protected void getPairFromRegistryServer( Integer id,
	                                          Object classOrMethod )
	{
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug(
			"getting pair " + id + ":" + classOrMethod + " from registry server, before sync" );
		}

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug(
			"getting pair " + id + ":" + classOrMethod + " from registry server, after sync" );
		}

		A.ssert( registryAddress_ != null || classOrMethod != null,
		         "Attempt to lookup unknown id " + id + " on registry server" );
		A.ssert( id != null || classOrMethod != null,
		         "Null id and classOrMethod" );

		if ( registryAddress_ == null )
		{
			addMapping( new Integer( nextID_++ ), classOrMethod );
		}
		else
		{
			//if we have multiple requests for the same ID on the server,
			//just wait until the first request finishes and send that answer
			Lock existingLock = null;
			Lock newLock = null;
			synchronized ( requestsInQueue_ )
			{
				Object requested = id == null ? classOrMethod : id;
				existingLock = (Lock) requestsInQueue_.get( requested );
				if ( existingLock == null )
				{
					newLock = new Lock();
					newLock.lock();
					requestsInQueue_.put( requested, newLock );
				}
			}

			//if there's already a thread asking for this id
			if ( existingLock != null )
			{
				//once the data has been retrieved for the existing lock,
				//our work is done
				existingLock.waitFor();
				return;
			}

			if ( id == null )
			{
				String name = getClassOrMethodName( classOrMethod );
				try
				{
					Registry r = getRegistry();
					ReturnValue o = r.getID( name );
					id = o.getInteger();
				}
				catch ( ARMIException ae )
				{
					LOG.error( ae.getMessage(), ae );
				}
			}
			else
			{
				if ( classOrMethod == null )
				{
					String classOrMethodString = "";
					try
					{
						Registry r = getRegistry();
						ReturnValue o = r.getClassOrMethod( id );
						classOrMethodString = o.getString();
					}
					catch ( ARMIException ae )
					{
						LOG.error( ae.getMessage(), ae );
					}

					classOrMethod = MethodMatcher.instance().
					classOrMethodForName( classOrMethodString );
				}
			}

			addMapping( id, classOrMethod );

			synchronized ( requestsInQueue_ )
			{
				requestsInQueue_.remove( newLock );
			}

			newLock.unlock();
		}
	}

	protected void addMapping( Integer id, Object classOrMethod, boolean saveToDisk )
	{
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Adding mapping from " + id + " to " + classOrMethod );
		}
		objectToIdMap_.put( classOrMethod, id );
		idToObjectMap_.put( id, classOrMethod );

		//if we load or otherwise obtain an id that is greater than or equal to our
		//current next id, bump the next id to be one higher
		if ( id.intValue() >= nextID_ )
		{
			nextID_ = id.intValue() + 1;
		}

		if ( saveToDisk )
		{
			saveMappings();
		}
	}

	protected void addMapping( Integer id, Object classOrMethod )
	{
		addMapping( id, classOrMethod, true );
	}

	public static final String MAP_FILE = "armi.idmap";

	public void saveMappings()
	{
		//TODO get idmap working more robustly (handle clients with old idmaps, mostly)
		//until then, don't save mappings

//        LOG.info("Saving ClassAndMethodTable to file");

//        BufferedWriter writer = null;

//        try {
//           writer = new BufferedWriter(new FileWriter(MAP_FILE));

//           synchronized(idToObjectMap_) {
//              Iterator iter = idToObjectMap_.keySet().iterator();
//              while(iter.hasNext()) {
//                 Object key = iter.next();
//                 Object value = idToObjectMap_.get(key);
//                 writer.write(key + ":" + getClassOrMethodName(value));
//                 writer.newLine();
//              }
//           }
//        } catch(Exception e) {
//           LOG.error("", e);
//        }
//        finally {
//           try {
//              if( writer != null ) writer.close();
//           } catch(Exception e) { LOG.error("", e); }
//        }

//        LOG.info("Done saving ClassAndMethodTable to file");
	}

	public void loadMappings()
	{
		//TODO get idmap working more robustly (handle clients with old idmaps, mostly)
		//until then, don't load mappings

//        if( new File(MAP_FILE).exists() )
//        {
//           LOG.info("Loading ClassAndMethodTable from file");

//           BufferedReader reader = null;

//           try
//           {
//              reader = new BufferedReader(new FileReader(MAP_FILE));

//              String line = null;
//              while( (line = reader.readLine()) != null )
//              {
//                 StringTokenizer toker = new StringTokenizer(line, ":");

//                 Integer key = Integer.valueOf(toker.nextToken());

//                 Object value = MethodMatcher.instance().
//                    classOrMethodForName(toker.nextToken());

//                 addMapping(key, value, false);
//              }
//           } catch(Exception e) {
//              LOG.error("", e);
//           }
//           finally {
//              try {
//                 if( reader != null ) reader.close();
//              } catch(Exception e) { LOG.error("", e); }
//           }

//           LOG.info("Done loading ClassAndMethodTable from file");
//        }
	}

	protected Registry getRegistry()
	{
		if ( registry_ == null )
		{
			registry_ = (Registry) ARMIProxyFactory.newInstance( Registry.class, registryAddress_.getIP() );
		}
		return registry_;
	}

	protected boolean registryShared_ = false;
	protected Map idToObjectMap_ = Collections.synchronizedMap( new HashMap() );
	protected Map objectToIdMap_ = new HashMap();
	protected int nextID_;
	protected int maxInfrastructureMethodID_;
	protected ServerAddress registryAddress_;
	protected Registry registry_;

	protected static ClassAndMethodTable instance_;

	static interface Registry
	{
		public ReturnValue /* Integer */ getID( String classOrMethod );

		public ReturnValue /*String */ getClassOrMethod( Integer id );
	}

	class ServerRegistry implements Registry
	{
		public ReturnValue getID( String classOrMethod )
		{
			//convert string into Class or Method object
			Object toMap = MethodMatcher.instance().
			classOrMethodForName( classOrMethod );
			//return id for Class or Method object
			ReturnValue rV = new ReturnValue( ClassAndMethodTable.this.getID( toMap ) );
			return rV;
		}

		public ReturnValue getClassOrMethod( Integer id )
		{
			ReturnValue v = new ReturnValue( ClassAndMethodTable.this.getName( id ) );
			return v;
		}

	}

	//debugging code follows
}
