/*
 * $Id: RMIMessageListener.java,v 1.1 2003/09/06 21:49:21 wurp Exp $
 */

package com.navtools.armi;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.navtools.armi.networking.ChannelMessenger;
import com.navtools.armi.networking.Message;
import com.navtools.armi.networking.MessageListener;
import com.navtools.util.PerformanceMonitor;
import com.navtools.util.VoidObject;
import org.apache.log4j.Category;
import org.apache.log4j.NDC;

public class RMIMessageListener implements MessageListener
{
	public static final Category LOG =
	Category.getInstance( RMIMessageListener.class.getName() );

	protected RMIMessageListener()
	{
	}

	public static RMIMessageListener instance()
	{
		if ( instance_ == null )
		{
			instance_ = new RMIMessageListener();
		}

		return instance_;
	}

	/**
	 * Start listening for remote method calls on obj.  You must call this
	 * method (normally via ARMI.share()) exactly once on each object you
	 * want to be accessible remotely.
	 * @param obj the object which will receive remote method calls
	 * @param interfaceImplemented the interface defining the calls on obj
	 * that can be made remotely
	 * @return an integer indicating the ID that obj will have on this ARMI
	 * server.  It is the number of objects implementing interfaceImplemented
	 * that have been shared so far on the server.  Any client who knows
	 * the IP and port of the server, the interfaceImplemented, and the ID
	 * can call methods on obj remotely.
	 */
	public int addInstance( Object obj, Class interfaceImplemented )
	{
		// check if shareable...
		List objList = (List) intrfaceMap_.get( interfaceImplemented );
		if ( objList == null )
		{
			objList = new ArrayList( 1 );
			intrfaceMap_.put( interfaceImplemented, objList );
		}
		objList.add( obj );

		int objIndex = objList.size() - 1;
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "TEST:registering instance " + objIndex + " implementing " +
			           interfaceImplemented );
		}

		ARMIProxyFactory.mapObjectToProxyInfo( obj,
		                                       ARMIProxyFactory.getMessenger().
		                                       getMessengerID(),
		                                       objIndex );
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "registered instance " + objIndex + " implementing " +
			           interfaceImplemented );
		}

		// temporary
		objToInterfaceMap_.put( obj, interfaceImplemented );

		return objIndex;
	}

	public boolean processMessage( Message msgArg )
	{
		if ( !( msgArg instanceof RMIMessage ) )
		{
			return false;
		}

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "RMIMessageListener about to process msg ID: " + msgArg.getMessageID() );
		}

		if ( ChannelMessenger.MSG_PERF.isDebugEnabled() )
		{
			ChannelMessenger.MSG_PERF.debug( "got process request for message id " +
			                                 msgArg.getMessageID() );
		}

		final RMIMessage msg = (RMIMessage) msgArg;

		if ( RMIMessageListener.LOG.isDebugEnabled() )
		{
			RMIMessageListener.LOG.debug(
			"RMIMessageListener processing msg ID: " + msg.getMessageID() );
		}

		processMessage( msg );

		return true;
	}

	public boolean processMessage( RMIMessage msg )
	{
		NDC.push( "processing msg " + msg.getMessageID() );
		try
		{
			if ( ChannelMessenger.MSG_PERF.isDebugEnabled() )
			{
				ChannelMessenger.MSG_PERF.debug( "processing message id " +
				                                 msg.getMessageID() );
			}

			long mcReconstructStart = System.currentTimeMillis(); // decide where to set this... and maybe check if we need to do it

			//determine method to invoke from method ID
			final int methodID = msg.getMethodID();
			final Method method = ClassAndMethodTable.instance().getMethod( new Integer( methodID ) );
			NDC.push( "Receiving " + method.getName() );
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "***Msg ID: " + msg.getMessageID() + ", remote call recvd***" );
			}

			//determine instance on which to invoke method from instance ID
			//and interface name
			final Object obj = getInstance( method.getDeclaringClass(),
			                                msg.getInstanceID() );
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "About to get method" );
			}

			//get list of arg types
			Class[] argTypes = method.getParameterTypes();

			final Object args[] =
			ARMIProxyFactory.readObjsFromMessage( msg, argTypes );

			if ( ChannelMessenger.MSG_PERF.isDebugEnabled() )
			{
				ChannelMessenger.MSG_PERF.debug( "invoking method " + method.getName() + " for message id " +
				                                 msg.getMessageID() );
			}

			//call method with object[] array
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "Invoking remote method" );
			}

			ReturnValue retVal = ReturnValue.NULL;
			try
			{
				retVal = (ReturnValue) method.invoke( obj, args );
			}
			catch ( Throwable e )
			{
				e.printStackTrace();

				StringBuffer errorMsg = new StringBuffer( "Throwable caught while invoking ARMI message " );
				errorMsg.append( method.getName() ).append( " with args " );

				if ( args != null )
				{
					for ( int i = 0; i < args.length; ++i )
					{
						errorMsg.append( args[i] );
					}
				}

				LOG.error( errorMsg.toString(), e );
				LOG.error( "Cause:", e.getCause() );
			}

			// before or after method call???
			if ( PerformanceMonitor.instance().isMcReconstructTimeEnabled() )
			{
				PerformanceMonitor.instance().addMcReconstructTime( new Long( msg.getClientKey().hashCode() + "" ),
				                                                    System.currentTimeMillis() - mcReconstructStart );
			}


			ClassAndMethodTable t = ClassAndMethodTable.instance();
			int returnTypeID;
			if ( retVal.getValue() == null )
			{
				returnTypeID = t.getID( VoidObject.class ).intValue();
			}
			else
			{
				if ( ARMIProxyFactory.isShared( retVal.getValue() ) )
				{
					returnTypeID = ( t.getID( (Class) objToInterfaceMap_.get( retVal.getValue() ) ) ).intValue();
				}
				else
				{
					returnTypeID = ( t.getID( retVal.getValue().getClass() ) ).intValue();
				}
			}


			long msgBuildStart = System.currentTimeMillis(); // decide where to set this... and maybe check if we need to do it

			RMIResponseMessage response =
			new RMIResponseMessage( Message.getNextID(),
			                        msg.getMessageID(),
			                        methodID, returnTypeID );

			if ( ( retVal.getValue() != null ) && ( retVal.getValue().getClass() != VoidObject.class ) )
			{
				ARMIProxyFactory.writeObjsToMessage( response,
				                                     new Object[]{retVal.getValue()},
				                                     new Class[]{retVal.getValue().getClass()} ); //unnecessary...
			}


			if ( PerformanceMonitor.instance().isMsgBuildTimeEnabled() )
			{
				PerformanceMonitor.instance().addMsgBuildTime( new Long( msg.getClientKey().hashCode() + "" ),
				                                               System.currentTimeMillis() - msgBuildStart );
			}

			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "***Msg ID: " + msg.getMessageID() + ", remote response sent***" );
			}

			/*
		   ARMIProxyFactory.
		      forObject(msg.getOrigin(), msg.getInstanceID()).
		      getOutgoingMessageQueue().send(response, msg.getOrigin());
		    */
			ARMIProxyFactory.getOutgoingMessageQueue().send( response, msg.getClientKey() );
		}
		catch ( Throwable e )
		{
			e.printStackTrace();
			LOG.error( "***Msg ID: " + msg.getMessageID() + " exception thrown***" );
			LOG.error( e.getMessage(), e );
		}
		finally
		{
			NDC.pop();
			NDC.pop();
		}

		return true;
	}

	public Object getInstance( Class intrface, int instanceID )
	{
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "getting instance " + instanceID + " implementing " + intrface );
		}
		List list = (List) intrfaceMap_.get( intrface );
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "list is: " + list );
		}
		return list.get( instanceID );
	}

	public boolean isShareable( Class c )
	{
		Method[] methods = c.getMethods();
		for ( int i = 0; i < methods.length; i++ )
		{
			if ( !( methods[i].getReturnType() == ReturnValue.class ) )
			{
				return false;
			}
		}
		return true;
	}

	protected Map intrfaceMap_ = new HashMap();
	protected Map objToInterfaceMap_ = new HashMap(); // temporary solution

	protected static RMIMessageListener instance_;

	//debugging code follows
	public static void main( String[] args ) throws Exception
	{
		/*TODO: change the shared instance to something other than
		 * ServerInterface; use a com.navtools.networking.armi.networking.test class that we don't change
		 ARMIProxyFactory.initializeAsRegistryServer();

		 RMIMessageListener.instance().addInstance(new ServerInterface()
		 {
		 public void logoff( long msgID, ObjectID objectId )
		 {
		 System.out.println("logoff called: " + msgID + ", " + objectId);
		 }

		 public void requestLogon( long msgID, String username, String characterName )
		 {
		 System.out.println("requestLogon called: " + msgID + ", " + username + ", " + characterName);
		 }


		 public void requestServerList(long msgID)
		 {
		 System.out.println("requestServerList called: " + msgID);
		 }


		 public void requestInventory(long msgID, ObjectID id)
		 {
		 System.out.println("requestInventory called: " + msgID + ", " + id);
		 }


		 public void init(long msgID, ClientInterface client, ServerAddress addy)
		 {
		 System.out.println("init called: " + msgID + ", " + client + ", " + addy);
		 }


		 public void shutdown(long msgID)
		 {
		 System.out.println("shutdown called: " + msgID);
		 }


		 public void updateObject(long msgID, ObjectID objectId, ObjectInformation info )
		 {
		 System.out.println("updateObject called: " + msgID + ", " + objectId + ", " + info);
		 }



		 public void pickUp(long msgID, ObjectID objectId )
		 {
		 System.out.println("pickUp called: " + msgID + ", " + objectId);
		 }


		 public void drop(long msgID, ObjectID objectId )
		 {
		 System.out.println("drop called: " + msgID + ", " + objectId);
		 }


		 public void sendTextMessage(long msgID,TextMessage message)
		 {
		 System.out.println("sendTextMessage called: " + msgID + ", " + message);
		 }


		 public void createCharacter(long msgID,String username, String characterName,
		 String shapeName, String textureName)
		 {
		 System.out.println("createCharacter called: " + msgID + ", " + username + ", " + characterName + ", " + shapeName + ", " + textureName);
		 }


		 public List getInstalledModels(long msgID)
		 {
		 System.out.println("getInstalledModels called: " + msgID);
		 return new ArrayList();
		 }


		 public List getInstalledTextures(long msgID)
		 {
		 System.out.println("getInstalledTextures called: " + msgID );
		 return new ArrayList();
		 }


		 public void getAllObjects(long msgID)
		 {
		 System.out.println("getAllObjects called: " + msgID);
		 }


		 public void putObjectIn(long msgID,ObjectID obj, ObjectID container)
		 {
		 System.out.println("putObjectIn called: " + msgID + ", " + obj + ", " + container);
		 }


		 public void attack(long msgID,ObjectID target)
		 {
		 System.out.println("attack called: " + msgID + ", " + target);
		 }

		 public void unrecognizedCommand(long msgID, String commandLine, ObjectID target)
		 {
		 System.out.println("unrecognizedCommand: " + msgID + ", " + commandLine + ", " + target);
		 }

		 },
		 ServerInterface.class);
		*/
	}
}

/*
  //on server
  MessageListener.addInstance(ServerProxy.instance());

//on client
MessageListener.addInstance(client);

//how to handle methods that must not be remote?
//standard method: build a local proxy wrapper that delegates remote methods to proxy

//to call ServerInterface methods from client
ServerProxy.instance().whatever();

ServerProxy.whatever()
{
remoteServer.whatever();
}

ProxyInstance remoteServer = UDPRemoteProxyInvocationHandler.getProxy(ServerInterface.class);
*/
