package com.navtools.armi;

import com.navtools.util.UnaryFunction;
import org.apache.log4j.Category;

/**
 * A wrapper class that encapsulates the value returned by a method of an ARMI shared object.
 * An instance of this class is returned if the object is remote. If it's local an instance
 * of ReturnValue is returned. The constructor of this class registers a handler with the
 * RMIResponseMessageListener and when the value comes back it's stored in the value variable
 * (declared in ReturnValue). The getter methods are used to get the value. If it hasn't been
 * received yet they wait for it to return.
 * @author Henrik Eriksson
 */
public class RemoteReturnValue extends ReturnValue
{
	public static final Category LOG =
	Category.getInstance( RemoteReturnValue.class.getName() );
	private long messageID;
	private boolean recieved;
	private Object lock = new Object();
	private UnaryFunction listener;

	public RemoteReturnValue( long msgID )
	{
		messageID = msgID;
		RMIResponseMessageListener.instance().addHandler( messageID, new UnaryFunction()
		{
			public Object execute( Object returnedValue )
			{
				value = returnedValue;
				recieved = true;
				synchronized ( lock )
				{
					lock.notifyAll();
				}
				valueReturned();
				return null;
			}
		} );
	}

	public long getMessageID()
	{
		return messageID;
	}

	public byte getbyte() throws ARMIException
	{
		try
		{
			return ( (Byte) getValue() ).byteValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public Byte getByte() throws ARMIException
	{
		try
		{
			return (Byte) getValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public boolean getboolean() throws ARMIException
	{
		try
		{
			return ( (Boolean) getValue() ).booleanValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public Boolean getBoolean() throws ARMIException
	{
		try
		{
			return (Boolean) getValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public char getchar() throws ARMIException
	{
		try
		{
			return ( (Character) getValue() ).charValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public Character getCharacter() throws ARMIException
	{
		try
		{
			return (Character) getValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public short getshort() throws ARMIException
	{
		try
		{
			return ( (Short) getValue() ).shortValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public Short getShort() throws ARMIException
	{
		try
		{
			return (Short) getValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public int getint() throws ARMIException
	{
		try
		{
			return ( (Integer) getValue() ).intValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public Integer getInteger() throws ARMIException
	{
		try
		{
			return (Integer) getValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public long getlong() throws ARMIException
	{
		try
		{
			return ( (Long) getValue() ).longValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public Long getLong() throws ARMIException
	{
		try
		{
			return (Long) getValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public float getfloat() throws ARMIException
	{
		try
		{
			return ( (Float) getValue() ).floatValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public Float getFloat() throws ARMIException
	{
		try
		{
			return (Float) getValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public double getdouble() throws ARMIException
	{
		try
		{
			return ( (Double) getValue() ).doubleValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public Double getDouble() throws ARMIException
	{
		try
		{
			return (Double) getValue();
		}
		catch ( Exception e )
		{
			throw new ARMIException( "Wrong return type. Should be " + value.getClass().getName() + "." );
		}
	}

	public String getString() throws ARMIException
	{
		Object retval = getValue();
		try
		{
			return (String) retval;
		}
		catch ( Exception e )
		{
		}
		return retval.toString();
	}

	public void onReturn( UnaryFunction listener )
	{
		this.listener = listener;
	}

	/**
	 * Called when the value comes back from the server; invokes the listener
	 */
	private void valueReturned()
	{
		if ( listener != null )
		{
			listener.execute( getValue() );
		}
	}

	/**
	 * This method combined with the constructor acts sort of like RMIResponseMessageListener.waitForReturn(...)
	 */
	public Object getValue()
	{
		synchronized ( lock )
		{
			while ( !recieved )
			{
				try
				{
					lock.wait(); //could block forever...
				}
				catch ( InterruptedException ie )
				{
					LOG.error( ie.getMessage(), ie );
				}
			}
		}

		return value;
	}
}
