/*
 * $Id: StandardUDPIncomingMessageQueue.java,v 1.2 2003/09/24 14:55:58 cactushack76 Exp $
 */

package com.navtools.armi.networking;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.channels.SelectionKey;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import com.navtools.armi.ARMIProxyFactory;
import com.navtools.armi.ClassAndMethodTable;
import com.navtools.armi.RMIMessage;
import com.navtools.armi.RMIResponseMessage;
import com.navtools.thread.CentralThreadPool;
import com.navtools.thread.Job;
import com.navtools.thread.Lock;
import com.navtools.thread.TaskMaster;
import com.navtools.util.A;
import com.navtools.util.Terminator;
import com.navtools.util.VoidFunction;
import org.apache.log4j.Category;

public class StandardUDPIncomingMessageQueue
{
	public static final Category LOG =
	Category.getInstance( StandardUDPIncomingMessageQueue.class.getName() );


	public static StandardUDPIncomingMessageQueue establishInstance()
	throws SocketException, UnknownHostException, Exception
	{
		return establishInstance( -1 );
	}


	public synchronized static StandardUDPIncomingMessageQueue establishInstance( int port )
	throws SocketException, UnknownHostException, Exception
	{
		A.ssert( instance_ == null,
		         "Attempt to start IMQ on port " + port + ", IMQ was already started" );

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Starting IMQ on port " + port );
		}

		instance_ = new StandardUDPIncomingMessageQueue( port );

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Created an instance of StandardUDPIncomingMessageQueue" );
		}

		return instance_;
	}


	public synchronized static StandardUDPIncomingMessageQueue establishInstance( InetAddress serverAddress, int port )
	throws SocketException, UnknownHostException, Exception
	{
                // There may be a case where we want to open a new connection to a different server
                if (instance_ != null && 
                    instance_.socket_ instanceof ChannelMessenger &&
                    ((ChannelMessenger)instance_.socket_).serverAddress_ != null && 
                    ((ChannelMessenger)instance_.socket_).serverAddress_.equals(serverAddress)) 
                {
		    A.ssert( instance_ == null,
		             "Attempt to start IMQ on port " + port + ", IMQ was already started" );
                }
                else if (instance_ != null) {
	       	    instance_.patientlyCloseAllComms();
                }

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Starting IMQ on port " + port );
		}

		instance_ = new StandardUDPIncomingMessageQueue( serverAddress, port );

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Created an instance of StandardUDPIncomingMessageQueue" );
		}

		return instance_;
	}


	protected StandardUDPIncomingMessageQueue( int port )
	throws SocketException, UnknownHostException, Exception
	{
		this( port == -1 ? ChannelMessenger.establish( ARMIProxyFactory.DEFAULT_PORT ) :
		      ChannelMessenger.establish( port ) );
	}


	protected StandardUDPIncomingMessageQueue( InetAddress serverAddress, int port )
	throws SocketException, UnknownHostException, Exception
	{
		this( port == -1 ? ChannelMessenger.establish( ARMIProxyFactory.DEFAULT_PORT ) :
		      ChannelMessenger.establishAndConnect( serverAddress, port, ARMIProxyFactory.DEFAULT_PORT ) );
	}


	protected StandardUDPIncomingMessageQueue( UDPMessengerInterface socket )
	throws SocketException
	{
		socket_ = socket;
		init();
		Terminator.instance().runOnExit( new VoidFunction()
		{
			public Object execute()
			{
				patientlyCloseAllComms();
                                instance_ = null;
				return null;
			}
		} );
	}


	public UDPMessengerInterface getMessenger()
	{
		return socket_;
	}


	protected StandardUDPIncomingMessageQueue()
	throws SocketException, UnknownHostException, Exception
	{
		this( -1 );
	}


	public static StandardUDPIncomingMessageQueue instance()
	{
		return instance_;
	}


	public int getLocalPort()
	{
		return socket_.getLocalPort();
	}


	public void init() throws SocketException
	{
//        new Thread(this, "StandardUDPIncomingMessageThread").start();
		socket_.setIncomingMessageQueue( this );
	}


	public void pushNewData( byte[] newData, SelectionKey key )
	{

		try
		{
			dp_.setLength( newData.length );
			dp_.setData( newData );
			dp_.setChannelKey( key );
			byte[] buffer = dp_.getData();

			ByteArrayInputStream bais = new ByteArrayInputStream( buffer );
			DataInputStream dis = new DataInputStream( bais );

			dis.mark( 100 );
			int messageType = dis.readInt();
			dis.reset();

			final Message msg = Message.buildMessage( messageType, dis );
			msg.setClientKey( dp_.getChannelKey() );


			Job processMessageJob = new Job()
			{
				public void execute()
				{
					try
					{
						processMessage( msg );
					}
					catch ( Exception e )
					{
						StandardUDPIncomingMessageQueue.LOG.error( e.getMessage(), e );
					}
				}
			};

			//Use a special thread to process ClassAndMethodTable messages, since they can
			//deadlock the system if not processed
			if ( isPriorityMessage( msg ) )
			{
				priorityPool_.execute( processMessageJob );
			}
			else
			{
				//must process message on another thread to avoid
				//clogging up messages received while processing this
				//message
				CentralThreadPool.instance().execute( processMessageJob );
			}
		}
		catch ( Exception e )
		{
			LOG.error( e.getMessage(), e );
			LOG.info( e.getMessage(), e );
		}
	}


	public void processMessage( final Message msg )
	{

		if ( ChannelMessenger.MSG_PERF.isDebugEnabled() )
		{
			ChannelMessenger.MSG_PERF.debug( "processing message id " + msg.getMessageID() );
		}


		Iterator iter = msgListeners_.iterator();
		boolean done = false;
		while ( iter.hasNext() && !done )
		{
			final MessageListener currListener = (MessageListener) iter.next();
			currListener.processMessage( msg );
		}
	}


	public void addMessageListener( MessageListener listener )
	{
		msgListeners_.add( listener );
	}


	public boolean isPriorityMessage( Message msg )
	{
		boolean priorityMessage = false;

		//first get the method the message is about
		int messageMethodID = -1;
		if ( msg instanceof RMIMessage )
		{
			RMIMessage rmsg = (RMIMessage) msg;
			messageMethodID = rmsg.getMethodID();

			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "RMIMessage incoming" );
			}
		}
		else
		{
			if ( msg instanceof RMIResponseMessage )
			{
				RMIResponseMessage rrmsg = (RMIResponseMessage) msg;
				messageMethodID = rrmsg.getMethodID();

				if ( LOG.isDebugEnabled() )
				{
					LOG.debug( "RMIResponseMessage incoming" );
				}
			}
		}

		//then com.navtools.networking.armi.networking.test to see if the the declaring class type is ClassAndMethodTable
		if ( messageMethodID != -1 )
		{
			priorityMessage = ClassAndMethodTable.instance().isInfrastructureMethod( messageMethodID );

			if ( LOG.isDebugEnabled() && priorityMessage )
			{
				LOG.debug( "Priority message, method call " + messageMethodID );
			}
		}

		return priorityMessage;
	}


	/**
	 * If processing received data, wait until processing is complete.
	 * Then close socket.
	 * Otherwise, close socket and return immediately.
	 */
	public static void patientlyCloseAllComms()
	{
		LOG.info( "Closing IMQ" );
		if ( !instance().listening_ )
		{
			Lock.logBeforeSync();
			synchronized ( instance().messageLoopLock_ )
			{
				Lock.logBeginSync();
				Lock.logEndSync();
			}
			Lock.logAfterSync();
		}
		instance().stop_ = true;
		instance().socket_.close();
	}


	public Set getClients()
	{
		return socket_.getClients();
	}


	public ChannelMessenger getSocket()
	{
		return (ChannelMessenger) socket_;
	}
        


	protected Object messageLoopLock_ = new Object();
	protected byte[] buffer_ = new byte[BUFFER_MAX];
	protected Packet dp_ = new Packet( buffer_, buffer_.length );
	protected UDPMessengerInterface socket_;
	protected boolean stop_ = false;
	protected boolean listening_ = false;
	protected List msgListeners_ = new LinkedList();
	protected TaskMaster priorityPool_ = new TaskMaster( 1, true );

	private static StandardUDPIncomingMessageQueue instance_;

	public final static int BUFFER_MAX = 8192;

	//com.navtools.networking.armi.networking.test code follows
	public static final int TEST_PORT = 4793;


	public static void main( String[] args ) throws Exception
	{
		//Tester.redirectOut("imq.out");

		StandardUDPIncomingMessageQueue.establishInstance( TEST_PORT );
		TestMessage.register();

		StandardUDPIncomingMessageQueue imq =
		StandardUDPIncomingMessageQueue.instance();

		imq.addMessageListener( new MessageListener()
		{
			public boolean processMessage( Message msg )
			{
				return true;
			}
		} );
	}
}
