package com.navtools.armi.networking;

import java.io.*;
import java.net.*;
import java.util.*;

import com.navtools.thread.BlockingQueue;
import com.navtools.thread.Job;
import com.navtools.thread.TaskMaster;
import com.navtools.util.ArrayUtil;
import com.navtools.util.Pair;
import com.navtools.util.PerformanceMonitor;
import org.apache.log4j.Category;
import java.nio.channels.SelectionKey;

public class TCPMessenger
implements UDPMessengerInterface
{
    public static final Category LOG =
    Category.getInstance( TCPMessenger.class.getName() );

    public static final Category MSG_PERF =
    Category.getInstance( "performance.networking" );

    protected TCPMessenger( int port )
    throws SocketException, UnknownHostException
    {
        messengerID_.setPort( port );

        if ( port != 0 )
        {
            messengerID_.setServerID( 0 );
        }

        messengerID_.setAddress( InetAddress.getLocalHost() );

        startServerSocket();
    }

    public static TCPMessenger establishInstance( int port )
    throws SocketException, UnknownHostException
    {
        //if establishInstance has already been called
        if ( instance_ != null )
        {
            //if the existing instance has a different port, throw an error
            //otherwise, just use the existing instance
            if ( instance_.getLocalPort() != port )
            {
                throw new Error( "TCPMessenger instance already established" );
            }
            else
            {
                return instance_;
            }
        }

        return instance_ = new TCPMessenger( port );
    }

    /**
     * This method is only here to work around a bug in InetAddress that
     * causes it to retrieve 127.0.0.1 for the local IP in some circumastances.
     * Set the local IP to the correct IP for remote access here.
     */
    public void setLocalAddress( InetAddress localAddress )
    {
        messengerID_.setAddress( localAddress );
    }

    public InetAddress getLocalAddress()
    {
        return messengerID_.getAddress();
    }

    public static TCPMessenger establishInstance()
    throws SocketException, UnknownHostException
    {
        //if establishInstance has already been called
        if ( instance_ != null )
        {
            //if the existing instance is not also just using the first
            //available port, throw an error.
            //otherwise, just use the existing instance
            if ( !instance_.isServer() )
            {
                throw new Error( "TCPMessenger instance already established" );
            }
            else
            {
                return instance_;
            }
        }

        return instance_ = new TCPMessenger( 0 );
    }

    public static TCPMessenger instance()
    {
        if ( instance_ == null )
        {
            try
            {
                establishInstance();
            }
            catch ( Exception e )
            {
                LOG.error( e.getMessage(), e );
            }
        }

        return instance_;
    }

    /**
     * Note that p must not be changed after this call is made; i.e. the packet should
     * be a temp object that is garbage collected after this call.
     */
    public void send( final Packet p )
    {
        if ( MSG_PERF.isDebugEnabled() )
        {
            MSG_PERF.debug( "Called send packet " + System.identityHashCode( p ) );
        }

        //do all networking on another thread.  The taskMaster_ has only one
        //thread, so all jobs put into him are automatically serialized.
        //(Not necessarily in the order in which you think you put them in,
        //since multiple threads may add jobs, but no two jobs will ever
        //be executed "at the same time")
        taskMaster_.execute( new Job()
        {
            public void execute()
            {
                DatagramSocketSend( p );
            }
        } );
    }

    public void receive( Packet p ) throws IOException
    {
        if ( LOG.isDebugEnabled() )
        {
            LOG.debug( "Waiting for incoming packet" );
        }

        Pair pair = (Pair) incomingMsgQueue_.get();

        MessengerID addy = (MessengerID) pair.getFirst();
        byte[] incoming = (byte[]) pair.getSecond();

        //fake out the datagrampacket to look as if it came
        //from the server that we got the data from.
        p.setMessengerID( addy );
        p.setData( incoming );
        p.setLength( incoming.length );
    }

    protected void DatagramSocketSend( Packet p )
    {
        SocketDescriptor sd = null;
        try
        {
            if ( MSG_PERF.isDebugEnabled() )
            {
                MSG_PERF.debug( "sending packet " + System.identityHashCode( p ) );
            }

            if ( LOG.isDebugEnabled() )
            {
                LOG.debug( "Before raw send" );
            }

            //wow, yuck.  Get the pair of in/output streams, then get
            //the output stream, then write the data buffer & length
            //to it.
            sd = getSocket( p.getMessengerID() );

            //don't try to send anything else to a disconnected socket
            if ( sd.isDisconnected() )
            {
                return;
            }

            if ( LOG.isDebugEnabled() )
            {
                LOG.debug( "Writing " + p.getLength() + " byte buffer to " +
                           sd.s.getInetAddress() + ":" + sd.s.getPort() );
            }

            if ( LOG.isDebugEnabled() )
            {
                LOG.debug( "Doublecheck outgoing data: " +
                           ArrayUtil.toHexString( p.getData(), 0, p.getLength() ) );
            }

            byte[] buff = p.getData();

            if ( buff.length != p.getLength() )
            {
                buff = new byte[p.getLength()];
                System.arraycopy( p.getData(), 0, buff, 0, p.getLength() );
            }

            sd.out.writeObject( buff );
            sd.out.flush();

            if ( MSG_PERF.isDebugEnabled() )
            {
                MSG_PERF.debug( "sent packet " + System.identityHashCode( p ) );
            }

            if ( LOG.isDebugEnabled() )
            {
                LOG.debug( "After raw send" );
            }

            if ( PerformanceMonitor.instance().isThroughputEnabled() )
            {
                PerformanceMonitor.instance().addOutgoingBytes( p.getMessengerID().getServerIDAsLong(),
                                                                System.currentTimeMillis(),
                                                                p.getLength() );
            }
        }
        catch ( IOException e )
        {
            LOG.warn( "", e );

            sd.disconnected( e );
        }
    }

    public int getLocalPort()
    {
        return messengerID_.getPort();
    }

    public void startServerSocket()
    {
        //note that per the serversocket docs, creating a serversocket on a
        //port of 0 opens it on any available port
        ServerSocket ss = null;

        try
        {
            ss = new ServerSocket( getLocalPort() );
        }
        catch ( IOException e )
        {
            LOG.error( "Error opening ServerSocket on port " + getLocalPort(),
                       e );
        }

        final ServerSocket serverSocket = ss;
        messengerID_.setPort( serverSocket.getLocalPort() );
        if ( LOG.isDebugEnabled() )
        {
            LOG.debug( "ServerSocket listening on port " + getLocalPort() );
        }

        Thread connectionListener = new Thread( "ConnectionListener" )
        {
            public void run()
            {
                while ( true )
                {
                    try
                    {
                        SocketDescriptor sd =
                        new SocketDescriptor( serverSocket.accept(),
                                              null );

                        LOG.debug( "Creating socket to accept client connection" );

                        if ( sd.s != null )
                        {
                            int port = sd.s.getPort();
                            InetAddress addy = sd.s.getInetAddress();

                            LOG.debug( "Read remote port " + port +
                                       " from socket" );

                            long serverID = random.nextLong();

                            //tell the client who it is
                            sd.out.writeLong( serverID );
                            sd.out.flush();

                            sd.messengerID = new MessengerID( addy, port,
                                                              serverID );
                            addMapping( sd.messengerID, sd );
                        }
                        else
                        {
                            LOG.error( "null socket returned by " +
                                       "serverSocket.accept()" );
                        }
                    }
                    catch ( IOException e )
                    {
                        LOG.error( "Exception in ServerSocket.accept", e );
                    }
                }
            }
        };

        connectionListener.start();
    }

    public void setIncomingMessageQueue( com.navtools.armi.networking.StandardUDPIncomingMessageQueue queue )
    {
        //this is just here for deprecation purposes
    }

    public void addMapping( final MessengerID messengerID,
                            final SocketDescriptor sd )
    {
        if ( LOG.isDebugEnabled() )
        {
            LOG.debug( "Adding mapping to " + messengerID + ", local port " +
                       sd.s.getLocalPort() );
        }

//      try
//      {
//          sd.s.setSoTimeout(2000);
//      }
//      catch(SocketException e)
//      {
//          LOG.error("Exception setting SoTimeout", e);
//      }

        new Thread( "Socket.receive" )
        {
            public void run()
            {
                try
                {
                    while ( true )
                    {
                        if ( MSG_PERF.isDebugEnabled() )
                        {
                            MSG_PERF.debug( "waiting for packet" );
                        }

                        byte[] incoming = (byte[]) sd.in.readObject();

                        if ( MSG_PERF.isDebugEnabled() )
                        {
                            MSG_PERF.debug( "received buffer " + System.identityHashCode( incoming ) );
                        }

                        if ( PerformanceMonitor.instance().isThroughputEnabled() )
                        {
                            PerformanceMonitor.instance().addIncomingBytes( messengerID.getServerIDAsLong(),
                                                                            System.currentTimeMillis(),
                                                                            incoming.length );
                        }

                        if ( LOG.isDebugEnabled() )
                        {
                            LOG.debug( "Received incoming packet" );

                            LOG.debug( "Size of incoming packet is " +
                                       incoming.length );
                            LOG.debug( "Doublecheck incoming data: " +
                                       ArrayUtil.toHexString( incoming, 0,
                                                              incoming.
                                                              length ) );
                        }

                        incomingMsgQueue_.add( new Pair( messengerID,
                                                         incoming ) );
                    }
                }
                catch ( SocketException e )
                {
                    if ( e.getMessage().indexOf( "Connection reset by peer" ) != -1 )
                    {
                        LOG.warn( "Lost connection to " + messengerID, e );
                    }
                    else
                    {
                        LOG.warn( "Couldn't read incoming byte[]", e );
                    }

                    sd.disconnected( e );
                }
                catch ( EOFException e )
                {
                    LOG.info( "Connection from " + messengerID +
                              " terminated normally", e );

                    sd.disconnected( e );
                }
                catch ( Exception e )
                {
                    LOG.warn( "Couldn't read incoming byte[]", e );

                    sd.disconnected( e );
                }
            }
        }.start();

        socketMap_.put( messengerID.getServerIDAsLong(), sd );
    }

    public void addDisconnectionListener(SelectionKey key, DisconnectionListener dcb)
    {
    }

    public void addDisconnectionListener( MessengerID messengerID,
                                          DisconnectionListener dcb )
    {
        getSocket( messengerID ).addDisconnectionListener( dcb );
    }

    public SocketDescriptor getSocket( MessengerID messengerID )
    {
        Long key = messengerID.getServerIDAsLong();

        if ( !socketMap_.containsKey( key ) )
        {
            try
            {
                //the server should never fail to look up the socket.
                //clients which have gotten a serverID shouldn't be
                //initiating outgoing connections.
                if ( messengerID_.getServerID() == 0 )
                {
                    LOG.error(
                    "Attempt made to make outgoing connection from server.  Server should already have a connection to anyone it should be talking to." );
                }

                //TODO handle client-to-client tunneling here if this
                //connection fails or connects to a different serverID
                //than expected
                SocketDescriptor sd =
                new SocketDescriptor( new Socket( messengerID.getAddress(),
                                                  messengerID.getPort() ),
                                      messengerID );

                if ( messengerID_.getServerID() == -1 )
                {
                    messengerID_.setServerID( sd.in.readLong() );
                }

                if ( LOG.isDebugEnabled() )
                {
                    LOG.debug( "Opening socket to server " +
                               messengerID.getServerAddress() );
                }

                addMapping( messengerID, sd );
            }
            catch ( IOException e )
            {
                LOG.error( "Failed to create socket to " +
                           messengerID.getServerAddress(), e );
            }
        }

        return (SocketDescriptor) socketMap_.get( key );
    }

    public void close()
    {
        LOG.info( "Close called; doing nothing" );
    }

    public boolean isServer()
    {
        return messengerID_.getServerID() == 0;
    }

    public MessengerID getMessengerID()
    {
        return messengerID_;
    }

    /**
     * Method that retrieves a Set of all the connected clients. Used for PerformanceMonitoring for now...
     */
    public Set getClients()
    {
        return socketMap_.keySet();
    }

    protected Map socketMap_ = Collections.synchronizedMap( new HashMap() );
    protected BlockingQueue incomingMsgQueue_ = new BlockingQueue();
    protected MessengerID messengerID_ = new MessengerID();

    protected static Random random = new Random();
    protected static TCPMessenger instance_;
    protected TaskMaster taskMaster_ = new TaskMaster( 1, false );


    public static class SocketDescriptor
    {
        public SocketDescriptor( Socket s,
                                 MessengerID messengerID ) throws IOException
        {
            out = s == null ? null :
                  new ObjectOutputStream( s.getOutputStream() );

            in = s == null ? null :
                 new ObjectInputStream( s.getInputStream() );

            realin = s == null ? null :
                     s.getInputStream();

            this.s = s;

            this.messengerID = messengerID;
        }

        public void addDisconnectionListener(SelectionKey key, DisconnectionListener dcb)
        {
        }

       public void addDisconnectionListener( DisconnectionListener dcb )
        {
            disconnectionListeners_.add( dcb );
        }

        public boolean isDisconnected()
        {
            return disconnected_;
        }

        public void disconnected( Exception e )
        {
            disconnected_ = true;

            Iterator iter = disconnectionListeners_.iterator();
            while ( iter.hasNext() )
            {
                ( (DisconnectionListener) iter.next() ).disconnected( e );
            }
        }

        ObjectOutputStream out;
        ObjectInputStream in;
        InputStream realin;
        Socket s;
        MessengerID messengerID;
        boolean disconnected_ = false;
        List disconnectionListeners_ = new ArrayList();
    }

    //com.navtools.networking.armi.networking.test code follows
    public static void main( String[] args ) throws Exception
    {
    }
}

/**
 Use cases:
 1) server receives incoming connection
 2) client tries to connect to server
 1) client checks to see if connection is registered for server
 2) if so, use that connection
 3) else
 1) if serverID == -1, throw an exception
 2) open TCP connection to the server's address & ip
 3) get serverID from server
 4) set serverID
 5) register the connection as 0
 3) client tries to connect to another client
 1) client checks to see if connection is registered for client
 2) if so, use that connection
 3) else
 1) if other client's serverID is -1, throw an exception
 2) client tries to connect to server
 3) client
 4) server tries to send a packet to a client
 1) if server doesn't have a connection registered for the client,
 throw an exception
 2) find the connection for that client.  Send packet.
 5) client tries to send a packet to a client
 6) peer receives a packet *handled*
 7) server receives an incoming direct connection request (extends 1)
 8) server receives an incoming pass-through connection request (extends 1)
 */
