/*
 * $Id: DataStreamableUtil.java,v 1.1 2003/09/06 21:49:22 wurp Exp $
 */

package com.navtools.serialization;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.*;

import com.navtools.armi.ClassAndMethodTable;
import com.navtools.util.A;
import com.navtools.util.MathUtil;
import com.navtools.util.Pair;
import org.apache.log4j.Category;

public class DataStreamableUtil
{
	public static final Category LOG =
	Category.getInstance( DataStreamableUtil.class.getName() );

	public static void setVerbose( boolean verbose )
	{
		verbose_ = verbose;
	}

	/**
	 * Writes out the contents of a list assuming that you
	 * will not know the exact type of the objects when you read them
	 * back.  You can write out a list of objects of various classes
	 * and read them back in with readList(DataInputStream dis)
	 */
	public static void writeList( DataOutputStream dos,
	                              List list )
	throws IOException
	{
		int size = list.size();
		dos.writeInt( size );

		//carefully write out exactly as many objects as we said the size
		//was in case of multithreading issues.  At least that will error
		//out right away worst case if there is a problem.
		Iterator iter = list.iterator();
		for ( int i = 0; i < size; ++i )
		{
			writeToForUnknownClass( iter.next(), dos );
		}
	}

	/**
	 * Reads in the contents of a list assuming that the
	 * objects have varied or unknown types.  Useful for reading in
	 * lists written out with writeList
	 */
	public static List readList( DataInputStream dis )
	throws IOException
	{
		int size = dis.readInt();
		List retval = new ArrayList( size );
		for ( int i = 0; i < size; ++i )
		{
			Object obj = null;
			try
			{
				obj = readFrom( dis );
			}
			catch ( Exception e )
			{
				LOG.error( "", e );
			}

			retval.add( obj );
		}

		return retval;
	}

	/**
	 * Writes out the contents of a list assuming that you
	 * will know the exact type of the objects when you read them
	 * back.  You can write out a list of object all the same class
	 * and read them back in with readList(DataInputStream dis, Class clazz)
	 */
	public static void writeListWithKnownClass( DataOutputStream dos,
	                                            List list )
	throws IOException
	{
		int size = list.size();
		dos.writeInt( size );

		//carefully write out exactly as many objects as we said the size
		//was in case of multithreading issues.  At least that will error
		//out right away worst case if there is a problem.
		Iterator iter = list.iterator();
		for ( int i = 0; i < size; ++i )
		{
			( (DataStreamable) iter.next() ).writeTo( dos );
		}
	}

	/**
	 * Reads in the contents of a list assuming that all
	 * objects have the exact type of clazz.  Useful for reading in
	 * lists written out with writeListWithKnownClass
	 */
	public static List readList( DataInputStream dis, Class clazz )
	throws IOException
	{
		int size = dis.readInt();
		List retval = new ArrayList( size );
		for ( int i = 0; i < size; ++i )
		{
			DataStreamable obj = null;
			try
			{
				obj = (DataStreamable) clazz.newInstance();
				obj.readFrom( dis );
			}
			catch ( Exception e )
			{
				LOG.error( "", e );
			}

			retval.add( obj );
		}

		return retval;
	}

	//corresponds with readFrom(DataInputStream in)
	//is useful if you won't know the Class on the receiving end
	public static void writeToForUnknownClass( Object toWrite,
	                                           DataOutputStream out )
	throws IOException
	{
		//get the class id from ClassAndMethodTable
		Class c = toWrite.getClass();
		int classId = ClassAndMethodTable.instance().getID( c ).intValue();

		//stream out the class id
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Writing out id " + classId + " for class " + c );
		}
		out.writeInt( classId );

		//stream out the instance data
		writeTo( toWrite, out );
	}

	//corresponds with writeToForUnknownClass(Object toWrite,
	//                                        DataInputStream in)
	//is useful if you won't know the Class on the receiving end
	public static Object readFrom( DataInputStream in )
	throws IOException
	{
		//stream in the class id
		int classId = in.readInt();

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Reading in class id " + classId );
		}

		//get the class from ClassAndMethodTable for that id
		Class c = ClassAndMethodTable.instance().getClass( new Integer( classId ) );

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Got class " + c + " for id " + classId );
		}

		return readFrom( c, in );
	}

	//corresponds with readFrom(Class theClass, DataInputStream in)
	//is only useful if you will know the Class on the receiving end
	public static void writeTo( Object toWrite, DataOutputStream out )
	throws IOException
	{
		A.ssert( toWrite != null, "Attempt to stream null object" );
		A.ssert( out != null, "Attempt to stream to null stream" );

		DataStreamer streamer = getStreamerFor( toWrite.getClass() );

		if ( streamer == null )
		{
			throw new NotDataStreamableException( toWrite.getClass().toString() );
		}
		else
		{
			if ( verbose_ )
			{
				verbose( "Writing object of type " + toWrite.getClass() );
			}

			streamer.writeTo( toWrite, out );
		}
	}

	//corresponds with writeTo(Object toWrite, DataInputStream in)
	//is useful if you will know the Class on the receiving end
	public static Object readFrom( Class theClass, DataInputStream in )
	throws IOException
	{
		A.ssert( theClass != null, "Attempt to read null class from stream" );
		A.ssert( in != null, "Attempt to read from null stream" );

		DataStreamer streamer = getStreamerFor( theClass );

		if ( streamer == null )
		{
			LOG.error( "Couldn't stream " + theClass );
			LOG.error( "", new NullPointerException() );
		}
		else
		{
			if ( verbose_ )
			{
				verbose( "Reading object of type " + theClass );
			}

			return streamer.readFrom( theClass, in );
		}

		return null;
	}

	protected static DataStreamer getStreamerFor( Class theClass )
	{
		DataStreamer streamer;

		if ( DataStreamable.class.isAssignableFrom( theClass ) )
		{
			streamer = StandardDataStreamer.instance();
		}
		else
		{
			streamer = (DataStreamer) getStreamerMap().get( theClass );
		}

		return streamer;
	}

	public static void addStreamer( Class toStream, DataStreamer streamer )
	{
		getStreamerMap().put( toStream, streamer );
	}

	protected static Map getStreamerMap()
	{
		if ( streamerMap_ == null )
		{
			streamerMap_ = new HashMap();
			//add some standard streamers
			DataStreamer longStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					Long retval = new Long( in.readLong() );
					if ( verbose_ )
					{
						verbose( "Reading long " + retval );
					}
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					if ( verbose_ )
					{
						verbose( "Writing long " + obj );
					}
					out.writeLong( ( (Long) obj ).longValue() );
				}
			};
			addStreamer( Long.class, longStreamer );
			addStreamer( long.class, longStreamer );

			DataStreamer intStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					Integer retval = new Integer( in.readInt() );
					if ( verbose_ )
					{
						verbose( "Reading int " + retval );
					}
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					if ( verbose_ )
					{
						verbose( "Writing int " + obj );
					}
					out.writeInt( ( (Integer) obj ).intValue() );
				}
			};
			addStreamer( Integer.class, intStreamer );
			addStreamer( int.class, intStreamer );

			DataStreamer shortStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					Short retval = new Short( in.readShort() );
					if ( verbose_ )
					{
						verbose( "Reading short " + retval );
					}
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					if ( verbose_ )
					{
						verbose( "Writing short " + obj );
					}
					out.writeShort( ( (Short) obj ).shortValue() );
				}
			};
			addStreamer( Short.class, shortStreamer );
			addStreamer( short.class, shortStreamer );

			DataStreamer byteStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					Byte retval = new Byte( in.readByte() );
					if ( verbose_ )
					{
						verbose( "Reading byte " + retval );
					}
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					if ( verbose_ )
					{
						verbose( "Writing byte " + obj );
					}
					out.writeByte( ( (Byte) obj ).byteValue() );
				}
			};
			addStreamer( Byte.class, byteStreamer );
			addStreamer( byte.class, byteStreamer );

			DataStreamer floatStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					Float retval = new Float( in.readFloat() );
					verbose( "Reading float " + retval );
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					verbose( "Writing float " + obj );
					out.writeFloat( ( (Float) obj ).floatValue() );
				}
			};
			addStreamer( Float.class, floatStreamer );
			addStreamer( float.class, floatStreamer );

			DataStreamer doubleStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					Double retval = new Double( in.readDouble() );
					verbose( "Reading double " + retval );
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					verbose( "Writing double " + obj );
					out.writeDouble( ( (Double) obj ).doubleValue() );
				}
			};
			addStreamer( Double.class, doubleStreamer );
			addStreamer( double.class, doubleStreamer );

			DataStreamer booleanStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					Boolean retval = new Boolean( in.readBoolean() );
					if ( verbose_ )
					{
						verbose( "Reading boolean " + retval );
					}
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					if ( verbose_ )
					{
						verbose( "Writing boolean " + obj );
					}
					out.writeBoolean( ( (Boolean) obj ).booleanValue() );
				}
			};
			addStreamer( Boolean.class, booleanStreamer );
			addStreamer( boolean.class, booleanStreamer );

			addStreamer( String.class, new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					String retval = new String( in.readUTF() );
					if ( verbose_ )
					{
						verbose( "Reading long " + retval );
					}
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					if ( verbose_ )
					{
						verbose( "Writing string " + obj );
					}
					out.writeUTF( (String) obj );
				}
			} );

			DataStreamer listStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					int listSize = in.readInt();

					if ( verbose_ )
					{
						verbose( "Reading list of size " + listSize );
					}

					List retval = new ArrayList( listSize );
					if ( listSize != 0 )
					{
						byte[] elementFlagBytes =
						new byte[( listSize * 2 + 7 ) / 8];
						in.read( elementFlagBytes );
						BitSet elementFlags =
						MathUtil.bytesToBitSet( elementFlagBytes,
						                        listSize * 2 );

						Class currClass = null;
						for ( int i = 0; i < listSize; ++i )
						{
							//if element is null
							if ( elementFlags.get( i * 2 ) )
							{
								if ( verbose_ )
								{
									verbose( "element " + i + " is null" );
								}
								retval.add( null );
							}
							else
							{
								//if is new class
								if ( elementFlags.get( i * 2 + 1 ) )
								{
									currClass =
									ClassAndMethodTable.instance().
									getClass( new Integer( in.readInt() ) );
									if ( verbose_ )
									{
										verbose( "element " + i + " is new class " + currClass );
									}
								}
								if ( verbose_ )
								{
									verbose( "reading element " + i );
								}
								Object elem =
								DataStreamableUtil.readFrom( currClass,
								                             in );
								retval.add( elem );
							}
						}
					}

					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					List list = new ArrayList( (List) obj );
					out.writeInt( list.size() );
					if ( verbose_ )
					{
						verbose( "Writing list of size " + list.size() );
					}

					if ( list.size() != 0 )
					{
						//build list of flags showing when class of
						//contained elements changes
						BitSet elementFlags = new BitSet( list.size() * 2 );
						Class lastClass = null;
						int i = 0;
						Iterator iter = list.iterator();
						while ( iter.hasNext() )
						{
							Object elem = iter.next();
							if ( elem == null )
							{
								elementFlags.set( i * 2 );
							}
							else
							{
								if ( !elem.getClass().equals( lastClass ) )
								{
									elementFlags.set( i * 2 + 1 );
								}
							}

							++i;
						}

						out.write( MathUtil.bitSetToBytes( elementFlags,
						                                   list.size() * 2 ) );

						iter = list.iterator();
						i = 0;
						lastClass = null;
						while ( iter.hasNext() )
						{
							Object elem = iter.next();
							//if element not null
							if ( !elementFlags.get( i * 2 ) )
							{
								//if new class
								if ( elementFlags.get( i * 2 + 1 ) )
								{
									lastClass = elem.getClass();
									out.writeInt( ClassAndMethodTable.instance().getID( lastClass ).intValue() );
									if ( verbose_ )
									{
										verbose( "element " + i + " is new class " + lastClass );
									}
								}
								if ( verbose_ )
								{
									verbose( "writing element " + i );
								}
								DataStreamableUtil.writeTo( elem, out );
							}
							else
							{
								if ( verbose_ )
								{
									verbose( "element " + i + " is null" );
								}
							}
							++i;
						}
					}
				}
			};
			addStreamer( List.class, listStreamer );
			addStreamer( ArrayList.class, listStreamer );
			addStreamer( LinkedList.class, listStreamer );
			addStreamer( Collections.EMPTY_LIST.getClass(), listStreamer );

			DataStreamer mapStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					int mapSize = in.readInt();
					Map retval = new HashMap( mapSize );

					if ( verbose_ )
					{
						verbose( "Reading map of size " + mapSize );
					}

					if ( mapSize != 0 )
					{
						byte[] elementFlagBytes =
						new byte[( mapSize * 4 + 7 ) / 8];
						in.read( elementFlagBytes );
						BitSet elementFlags =
						MathUtil.bytesToBitSet( elementFlagBytes,
						                        mapSize * 4 );

						Class currKeyClass = null;
						Class currValueClass = null;
						Object key = null;
						Object value = null;
						for ( int i = 0; i < mapSize; ++i )
						{
							//if key is null
							if ( elementFlags.get( i * 4 ) )
							{
								if ( verbose_ )
								{
									verbose( "key " + i + " is null" );
								}
								key = null;
							}
							else
							{
								//if key is new class
								if ( elementFlags.get( i * 4 + 1 ) )
								{
									currKeyClass =
									ClassAndMethodTable.instance().
									getClass( new Integer( in.readInt() ) );
									if ( verbose_ )
									{
										verbose( "key " + i + " is new class " + currKeyClass );
									}
								}
								if ( verbose_ )
								{
									verbose( "reading key " + i );
								}
								key = DataStreamableUtil.
								readFrom( currKeyClass, in );
							}

							//if value is null
							if ( elementFlags.get( i * 4 + 2 ) )
							{
								if ( verbose_ )
								{
									verbose( "value " + i + " is null" );
								}
								value = null;
							}
							else
							{
								//if value is new class
								if ( elementFlags.get( i * 4 + 3 ) )
								{
									currValueClass =
									ClassAndMethodTable.instance().
									getClass( new Integer( in.readInt() ) );
									if ( verbose_ )
									{
										verbose( "value " + i + " is new class " + currValueClass );
									}
								}
								if ( verbose_ )
								{
									verbose( "reading value " + i );
								}
								value = DataStreamableUtil.
								readFrom( currValueClass, in );
							}
							retval.put( key, value );
						}
					}

					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					Map map = new HashMap( (Map) obj );
					out.writeInt( map.size() );

					if ( verbose_ )
					{
						verbose( "Writing map of size " + map.size() );
					}

					if ( map.size() != 0 )
					{
						//build list of flags showing when class of
						//contained keys or values change
						BitSet elementFlags = new BitSet( map.size() * 4 );
						Class lastKeyClass = null;
						Class lastValueClass = null;
						int i = 0;
						Set entrySet = map.entrySet();
						Iterator iter = entrySet.iterator();
						while ( iter.hasNext() )
						{
							Map.Entry entry = (Map.Entry) iter.next();
							Object key = entry.getKey();
							Object value = entry.getValue();

							if ( key == null )
							{
								elementFlags.set( i * 4 );
							}
							else
							{
								if ( !key.getClass().equals( lastKeyClass ) )
								{
									elementFlags.set( i * 4 + 1 );
								}
							}

							if ( value == null )
							{
								elementFlags.set( i * 4 + 2 );
							}
							else
							{
								if ( !value.getClass().equals( lastValueClass ) )
								{
									elementFlags.set( i * 4 + 3 );
								}
							}

							++i;
						}

						out.write( MathUtil.bitSetToBytes( elementFlags,
						                                   map.size() * 4 ) );

						iter = entrySet.iterator();
						i = 0;
						lastKeyClass = null;
						lastValueClass = null;
						while ( iter.hasNext() )
						{
							Map.Entry entry = (Map.Entry) iter.next();
							Object key = entry.getKey();
							Object value = entry.getValue();
							//if key not null
							if ( !elementFlags.get( i * 4 ) )
							{
								//if new class
								if ( elementFlags.get( i * 4 + 1 ) )
								{
									lastKeyClass = key.getClass();
									if ( verbose_ )
									{
										verbose( "key " + i + " is new class " + lastKeyClass );
									}
									out.writeInt( ClassAndMethodTable.instance().getID( lastKeyClass ).intValue() );
								}
								if ( verbose_ )
								{
									verbose( "writing key " + i );
								}
								DataStreamableUtil.writeTo( key, out );
							}
							else
							{
								if ( verbose_ )
								{
									verbose( "key " + i + " is null" );
								}
							}

							//if value not null
							if ( !elementFlags.get( i * 4 + 2 ) )
							{
								//if new class
								if ( elementFlags.get( i * 4 + 3 ) )
								{
									lastValueClass = value.getClass();
									out.writeInt( ClassAndMethodTable.instance().getID( lastValueClass ).intValue() );
									if ( verbose_ )
									{
										verbose( "value " + i + " is new class " + lastValueClass );
									}
								}
								if ( verbose_ )
								{
									verbose( "writing value " + i );
								}
								DataStreamableUtil.writeTo( value, out );
							}
							else
							{
								if ( verbose_ )
								{
									verbose( "value " + i + " is null" );
								}
							}
							++i;
						}
					}
				}
			};
			addStreamer( Map.class, mapStreamer );
			addStreamer( HashMap.class, mapStreamer );
			addStreamer( Collections.EMPTY_MAP.getClass(), mapStreamer );

			DataStreamer pairStreamer = new DataStreamer()
			{
				public Object readFrom( Class theClass, DataInputStream in )
				throws IOException
				{
					Pair retval = new Pair( null, null );
					byte[] elementFlagBytes =
					new byte[1];
					in.read( elementFlagBytes );
					BitSet elementFlags =
					MathUtil.bytesToBitSet( elementFlagBytes,
					                        3 );

					Class currClass = null;
					//if first element is null
					if ( elementFlags.get( 0 ) )
					{
						retval.setFirst( null );
					}
					else
					{
						currClass = ClassAndMethodTable.instance().
						getClass( new Integer( in.readInt() ) );
						retval.setFirst( DataStreamableUtil.
						                 readFrom( currClass, in ) );
					}
					//if second element is null
					if ( elementFlags.get( 1 ) )
					{
						retval.setSecond( null );
					}
					else
					{
						//if second element has different class
						//than first element
						if ( elementFlags.get( 2 ) )
						{
							currClass = ClassAndMethodTable.instance().
							getClass( new Integer( in.readInt() ) );
						}
						retval.setSecond( DataStreamableUtil.
						                  readFrom( currClass, in ) );
					}

					if ( verbose_ )
					{
						verbose( "read pair(" + retval.getFirst() == "null" ?
						         null : retval.getFirst().getClass() + ", " +
						                retval.getSecond() == null ?
						                "null" : retval.getSecond().getClass() + ")" );
					}
					return retval;
				}

				public void writeTo( Object obj, DataOutputStream out )
				throws IOException
				{
					Pair p = (Pair) obj;

					if ( verbose_ )
					{
						verbose( "writing pair(" + p.getFirst() == "null" ?
						         null : p.getFirst().getClass() + ", " +
						                p.getSecond() == null ?
						                "null" : p.getSecond().getClass() + ")" );
					}

					//build flags showing when class of contained
					//elements changes and which elems are null
					BitSet elementFlags = new BitSet( 3 );
					Class lastClass = null;
					if ( p.getFirst() == null )
					{
						elementFlags.set( 0 );
					}
					else
					{
						lastClass = p.getFirst().getClass();
					}

					if ( p.getSecond() == null )
					{
						elementFlags.set( 1 );
					}
					else
					{
						if ( !p.getSecond().getClass().equals( lastClass ) )
						{
							elementFlags.set( 2 );
						}
					}

					out.write( MathUtil.bitSetToBytes( elementFlags, 3 ) );

					//if first element is not null
					if ( !elementFlags.get( 0 ) )
					{
						out.writeInt( ClassAndMethodTable.instance().getID( p.getFirst().getClass() ).intValue() );
						DataStreamableUtil.writeTo( p.getFirst(), out );
					}
					//if second element is not null
					if ( !elementFlags.get( 1 ) )
					{
						//if second element has different class than first
						if ( elementFlags.get( 2 ) )
						{
							out.writeInt( ClassAndMethodTable.instance().getID( p.getSecond().getClass() ).intValue() );
						}
						DataStreamableUtil.writeTo( p.getSecond(), out );
					}
				}
			};
			addStreamer( Pair.class, pairStreamer );
		}

		return streamerMap_;
	}

	public static void verbose( String toWrite )
	{
		if ( verbose_ )
		{
			LOG.info( toWrite );
		}
	}

	protected static Map streamerMap_;
	protected static boolean verbose_ = false;
}

class StandardDataStreamer implements DataStreamer
{
	public Object readFrom( Class theClass, DataInputStream in )
	throws IOException
	{
		Object retval = null;

		try
		{
			retval = theClass.newInstance();
		}
		catch ( Exception e )
		{
			LOG.error( "Unable to read " + theClass.getName() +
			           " from stream.  Failure in call to " +
			           "public default constructor." );
			LOG.error( e.getMessage(), e );
		}

		try
		{
			( (DataStreamable) retval ).readFrom( in );
		}
		catch ( Exception e )
		{
			LOG.error( e.getMessage(), e );
		}

		return retval;
	}

	public void writeTo( Object obj, DataOutputStream out ) throws IOException
	{
		( (DataStreamable) obj ).writeTo( out );
	}

	protected static StandardDataStreamer instance()
	{
		if ( instance_ == null )
		{
			instance_ = new StandardDataStreamer();
		}

		return instance_;
	}

	protected static StandardDataStreamer instance_;
}
