package com.navtools.util;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

/**
 * A FIFO queue with which you can read and write bytes.  Provides efficient support
 * for reading an arbitrary number of bytes from a channel and storing them until
 * you are ready to use them.
 */
public class ByteQueue
{
	/**
	 * This class uses a ring of ByteBuffers to hold the data.  In each ByteBuffer,
	 * the limit is used as the position from which to start writing and at which to stop reading,
	 * the position is used as the position at which to start reading, and the capacity is the
	 * position at which to stop writing.
	 *
	 * One buffer in the ring is the current read buffer, one is the current write buffer.  When the
	 * position reaches the limit in the read buffer, and the limit != capacity, then there is no more
	 * data to be read.  When the position reaches the limit in the read buffer and limit == capacity, then
	 * if the read buffer is incremented to the next buffer and we try again to read from there.
	 *
	 * When the capacity reaches the limit in the write buffer, we check to see if the next buffer is
	 * the read buffer.  If it is, then we create a new buffer and insert it between the write buffer and
	 * the read buffer.  If not, then we increment the write buffer (again, wrapping if necessary).
	 *
	 * Every buffer must be one of: the read buffer, the write buffer, a full buffer, or an empty buffer.
	 * The read buffer and the write buffer may be the same.  For the read buffer, the position is where to begin
	 * reading and the limit is where to stop reading.  For the write buffer the limit is where to start writing
	 * and the capacity is the end of the buffer.  For the full buffers, the position is 0 and the limit is
	 * the capacity.  For the empty buffers, the position is 0 and the limit is 0.
	 */
	//size of ByteBuffers
	private int buffSize;

	//build a ring of ByteBuffers
	private ByteBuffer[] bbRing;

	//keep track of BB in ring to which data is written.
	private int writeBB;

	//keep track of last BB in ring from which data is read. From the write BB up to the read BB, no data is currently stored.
	private int readBB;

	public ByteQueue()
	{
		this( 1, 1024 );
	}

	public ByteQueue( int numBuffs, int buffSize )
	{
		this.buffSize = buffSize;

		bbRing = new ByteBuffer[numBuffs];
		for ( int i = 0; i < numBuffs; ++i )
		{
			bbRing[i] = allocateNewBuffer();
		}
	}

	public int getBuffSize()
	{
		return buffSize;
	}

	public void push( byte[] bytes )
	{
		push( bytes, 0, bytes.length );
	}

	public void push( byte[] bytes, int begin, int length )
	{
		synchronized ( this )
		{
			int numBytesWritten = 0;
			while ( length > numBytesWritten )
			{
				//if at end of buffer, move to next buffer
				if ( getWriteBuff().limit() >= getWriteBuff().capacity() )
				{
					incrementWriteBuff();
				}

				int oldPosition = getWriteBuff().position();

				//since the put operation starts at the position and stops at limit, set the position to the old limit and the limit to the capacity
				getWriteBuff().position( getWriteBuff().limit() );
				getWriteBuff().limit( getWriteBuff().capacity() );

				int numBytesToWrite = Math.min( getWriteBuff().remaining(), length - numBytesWritten );
				getWriteBuff().put( bytes, begin + numBytesWritten, numBytesToWrite );
				numBytesWritten += numBytesToWrite;

				getWriteBuff().limit( getWriteBuff().position() );
				getWriteBuff().position( oldPosition );
			}
		}
	}

	public void push( int b )
	{
		synchronized ( this )
		{
			push( new byte[]{(byte) b} );
		}
	}

	/**
	 * Push all bytes available to read from the given channel onto the queue.
	 */
	public void push( ReadableByteChannel chann ) throws IOException
	{
		synchronized ( this )
		{
			int numBytesLastWritten = 0;
			do
			{
				//if at end of buffer, move to next buffer
				if ( getWriteBuff().limit() >= getWriteBuff().capacity() )
				{
					incrementWriteBuff();
				}

				int oldPosition = getWriteBuff().position();

				//since the put operation starts at the position and stops at limit, set the position to the old limit and the limit to the capacity
				getWriteBuff().position( getWriteBuff().limit() );
				getWriteBuff().limit( getWriteBuff().capacity() );

				numBytesLastWritten = chann.read( getWriteBuff() );

				getWriteBuff().limit( getWriteBuff().position() );
				getWriteBuff().position( oldPosition );
			}
			while ( numBytesLastWritten > 0 );
		}
	}

	/**
	 * @return the number of bytes remaining for peek or pop.
	 */
	public int remaining()
	{
		int retval = 0;

		synchronized ( this )
		{
			if ( readBB == writeBB )
			{
				retval = getReadBuff().remaining();
			}
			else
			{
				if ( writeBB > readBB )
				{
					retval = getReadBuff().remaining() + getWriteBuff().remaining() + buffSize * ( writeBB - readBB - 1 );
				}
				else
				{
					retval = getReadBuff().remaining() + getWriteBuff().remaining() + buffSize * ( bbRing.length - readBB + writeBB - 1 );
				}
			}

			debug( "Calculated remaining" );
			printRingDebugInfo();
			debug( "Remaining is " + retval );
		}

		return retval;
	}

	/**
	 * Pop all remaining bytes from the queue.
	 */
	public byte[] pop()
	{
		byte[] retval;
		synchronized ( this )
		{
			retval = new byte[remaining()];
			pop( retval );
		}

		return retval;
	}

	/**
	 * Pop up to buffer.length remaining bytes from the queue.  Put the bytes into buffer.
	 * @return The number of bytes popped.
	 */
	public int pop( byte[] buffer )
	{
		return pop( buffer, 0, buffer.length );
	}

	/**
	 * Pop up to length remaining bytes from the queue, putting them into
	 * buffer, starting at begin.
	 * @return The number of bytes popped.
	 */
	public int pop( byte[] buffer, int begin, int length )
	{
		return peekOrPop( buffer, begin, length, false );
	}

	/**
	 * Pop as many bytes from as the channel will accept.  Write those bytes on the channel.
	 * @return The number of bytes popped.
	 */
	public int pop( WritableByteChannel channel ) throws IOException
	{
		int numBytesWritten = 0;
		int lastNumBytesWritten;
		int lastNumBytesRemaining;

		synchronized ( this )
		{
			do
			{
				if ( getReadBuff().remaining() == 0 )
				{
					incrementReadBuff();
				}

				lastNumBytesRemaining = getReadBuff().remaining();
				lastNumBytesWritten = channel.write( getReadBuff() );
				numBytesWritten += lastNumBytesWritten;

				//continue until we weren't able to write out all of the bytes or until we have no more data to write
			}
			while ( ( lastNumBytesWritten == lastNumBytesRemaining ) && ( remaining() > 0 ) );
		}

		return numBytesWritten;
	}

	/**
	 * Put up to buffer.length remaining bytes from the queue into buffer without removing them from the queue.
	 * @return The number of bytes put into buffer.
	 */
	public int peek( byte[] buffer )
	{
		return peek( buffer, 0, buffer.length );
	}

	/**
	 * Put up to length remaining bytes from the queue into buffer starting at begin, without removing the bytes from the queue.
	 * @return The number of bytes put into buffer.
	 */
	public int peek( byte[] buffer, int begin, int length )
	{
		debug( "before peek" );
		printRingDebugInfo();
		int retval = peekOrPop( buffer, begin, length, true );
		printRingDebugInfo();
		debug( "after peek" );

		return retval;
	}

	/**
	 * Look at all remaining bytes on the queue without removing them from the queue.
	 */
	public byte[] peek()
	{
		byte[] buffer;

		synchronized ( this )
		{
			buffer = new byte[remaining()];
			peek( buffer );
		}

		return buffer;
	}

	public int peekOrPop( byte[] buffer, int begin, int length, boolean isPeek )
	{
		int numBytesRead = 0;
		int numBytesToRead = 0;
		int totalBytesToRead = Math.min( length, remaining() );
		debug( "Begin peekOrPop, reading " + length );

		synchronized ( this )
		{
			//store off original read buff.  Only needed to restore state after a peek operation.
			int originalReadBB = -1;
			if ( isPeek )
			{
				originalReadBB = readBB;
			}

			//variable for read buff state.  Only needed to restore state after a peek operation.
			int originalPosition = -1;
			int originalLimit = -1;

			while ( numBytesRead < totalBytesToRead )
			{
				if ( getReadBuff().remaining() == 0 )
				{
					ByteBuffer oldReadBuff = getReadBuff();

					incrementReadBuff();

					//if we're peeking and this isn't the first time through the loop, recover the buffer position
					if ( isPeek && originalPosition != -1 && originalLimit != -1 )
					{
						oldReadBuff.limit( originalLimit );
						oldReadBuff.position( originalPosition );
					}
				}

				//save the read buffer data so we can recover
				if ( isPeek )
				{
					originalPosition = getReadBuff().position();
					originalLimit = getReadBuff().limit();
				}

				numBytesToRead = Math.min( getReadBuff().remaining(), totalBytesToRead - numBytesRead );
				try
				{
					getReadBuff().get( buffer, begin + numBytesRead, numBytesToRead );
				}
				catch ( RuntimeException e )
				{
					debug(
					"Attempting to populate buffer of length " + buffer.length + " starting at " + ( begin + numBytesRead ) + ", length " + numBytesToRead );
					debug( "from a ByteBuffer with " + getReadBuff().remaining() + " bytes available" );
					throw e;
				}

				numBytesRead += numBytesToRead;
			}

			//if we're peeking and we entered the loop, fix the buffer state
			if ( isPeek && originalPosition != -1 && originalLimit != -1 )
			{
				getReadBuff().position( originalPosition );
				getReadBuff().limit( originalLimit );
			}

			if ( isPeek )
			{
				readBB = originalReadBB;
			}
		}

		return numBytesRead;
	}

	private ByteBuffer getReadBuff()
	{
		return bbRing[readBB];
	}

	private ByteBuffer getWriteBuff()
	{
		return bbRing[writeBB];
	}

	private void incrementWriteBuff()
	{
		writeBB = ( writeBB + 1 ) % bbRing.length;
		synchronized ( this )
		{
			if ( writeBB == readBB )
			{
				debug( "Expanding" );
				printRingDebugInfo();

				ByteBuffer[] oldBBRing = bbRing;
				bbRing = new ByteBuffer[(int) ( bbRing.length * 1.5 + 1 )];
				int sizeDiff = bbRing.length - oldBBRing.length;
				readBB += sizeDiff;

				//copy the buffers before writeBB to the same position in the new ring
				System.arraycopy( oldBBRing, 0, bbRing, 0, writeBB );

				debug( "Inserting new buffers between " + sizeDiff + " and " + ( sizeDiff + writeBB ) );
				//put new buffers between the writeBB and readBB
				for ( int i = 0; i < sizeDiff; ++i )
				{
					bbRing[i + writeBB] = allocateNewBuffer();
				}

				//put the tail end of the oldBBRing at the end of bbRing
				System.arraycopy( oldBBRing, writeBB, bbRing, writeBB + sizeDiff, oldBBRing.length - writeBB );

				printRingDebugInfo();
				debug( "Done expanding" );
			}
		}
	}

	/**
	 * Increment to the next read buffer, setting the limit on the last read buffer to 0
	 * so it can be re-used for writing.
	 */
	private void incrementReadBuff()
	{
		if ( readBB == writeBB )
		{
			throw new RuntimeException( "Buffer underflow" );
		}
		//reset the buffer to be available for writing
		getReadBuff().limit( 0 );
		readBB = ( readBB + 1 ) % bbRing.length;
	}

	private ByteBuffer allocateNewBuffer()
	{
		ByteBuffer retval = ByteBuffer.allocate( buffSize );
		retval.limit( 0 );
		return retval;
	}

	private void printRingDebugInfo()
	{
		if ( isDebugEnabled() )
		{
			debug( "Current read " + readBB + " current write " + writeBB );
			for ( int i = 0; i < bbRing.length; ++i )
			{
				debug( "buffer " + i + " has " + bbRing[i].remaining() + " bytes available to read" );
				debug( "buffer " + i + " has " + ( bbRing[i].capacity() - bbRing[i].limit() ) + " bytes available to write" );
			}
		}
	}

	private void debug( String msg )
	{
		if ( isDebugEnabled() )
		{
			System.out.println( msg );
		}
	}

	private boolean isDebugEnabled()
	{
		return false;
	}
}
