/*
 * $Id: MathUtil.java,v 1.4 2003/09/06 21:49:22 wurp Exp $
 */

package com.navtools.util;

import java.util.BitSet;
import java.util.Random;

import org.apache.log4j.Category;

public class MathUtil
{
	public static final Category LOG =
	Category.getInstance( MathUtil.class.getName() );

	public static final double PI_2 = Math.PI / 2;

	/**
	 * Converts a BitSet to a compact sequence of bytes that contain
	 * the bits in the BitSet.  Can convert back into a BitSet by
	 * calling bytesToBitSet.
	 */
	public static byte[] bitSetToBytes( BitSet bs, int numBits )
	{
		byte[] retval = new byte[( numBits + 7 ) / 8];
		byte orVal = 1;
		for ( int i = 0; i < numBits; ++i )
		{
			if ( i % 8 == 0 )
			{
				orVal = 1;
			}
			if ( bs.get( i ) )
			{
				retval[i / 8] |= orVal;
			}
			orVal <<= 1;
		}

		return retval;
	}

	public static BitSet bytesToBitSet( byte[] bytes, int numBits )
	{
		BitSet retval = new BitSet( numBits );
		byte orVal = 1;
		for ( int i = 0; i < numBits; ++i )
		{
			if ( i % 8 == 0 )
			{
				orVal = 1;
			}
			if ( ( bytes[i / 8] & orVal ) != 0 )
			{
				retval.set( i );
			}
			orVal <<= 1;
		}

		return retval;
	}

	/**
	 * Returns a number with the same sign as num.  Will only return
	 * -1, 0, or 1
	 */
	public static short sign( double num )
	{
		return ( num < 0 ) ? (short) -1 : ( ( num > 0 ) ? (short) 1 : (short) 0 );
	}

	/**
	 * Solves quadratic equations of the form a*x^2 + b*x + c.
	 * @param a the factor of x^2
	 * @param b the factor of x
	 * @param c the constant
	 * @returns double[] array of solutions (2 elements)
	 */
	public double[] quadraticSolutions( double a, double b, double c )
	{
		double bFactor = -b / 2 * a;
		double otherFactor = ( b * b - 4 * a * c ) / 2 * a;
		double[] solutions = new double[]{bFactor + otherFactor,
		                                  bFactor - otherFactor};

		return solutions;
	}

	/**
	 * If num is within tolerance of an integer, the integer is returned.
	 * Otherwise, num is returned.
	 */
	public static double snap( double num, double tolerance )
	{
		if ( Math.abs( Math.round( num ) - num ) < tolerance )
		{
			return Math.round( num );
		}
		else
		{
			return num;
		}
	}

	/**
	 * If num is within tolerance of an integer, the integer is returned.
	 * Otherwise, num is returned.
	 */
	public static float snap( float num, float tolerance )
	{
		if ( Math.abs( Math.round( num ) - num ) < tolerance )
		{
			return Math.round( num );
		}
		else
		{
			return num;
		}
	}

	/**
	 * Retrieves an integer as 4 bytes in a buffer, with the first byte
	 * being at index start.
	 * Uses the standard serialization format for an integer.
	 */
	public static int asInt( byte[] buffer, int start )
	{
		return ( ( buffer[start] & 0xff ) << 24 ) +
		       ( ( buffer[start + 1] & 0xff ) << 16 ) +
		       ( ( buffer[start + 2] & 0xff ) << 8 ) +
		       ( buffer[start + 3] & 0xff );
	}

	/**
	 *  Takes an integer and returns a byte array representing that integer.
	 */

	public static byte[] asBytes( int i )
	{
		byte[] bytes = new byte[4];
		bytes[0] = (byte) ( ( i >> 24 ) & 0xff );
		bytes[1] = (byte) ( ( i >> 16 ) & 0xff );
		bytes[2] = (byte) ( ( i >> 8 ) & 0xff );
		bytes[3] = (byte) ( ( i >> 0 ) & 0xff );
		return bytes;
	}

	public static boolean equalsTolerance( double num1, double num2,
	                                       double tolerance )
	{
		return Math.abs( num1 - num2 ) < tolerance;
	}

	public static boolean equalsTolerance( float num1, float num2,
	                                       float tolerance )
	{
		return Math.abs( num1 - num2 ) < tolerance;
	}

	/**
	 * If num is within tolerance of snapNum, snapNum is returned.
	 * Otherwise, num is returned.
	 */
	public static double snapTo( double num, double tolerance, double snapNum )
	{
		if ( Math.abs( snapNum - num ) < tolerance )
		{
			return snapNum;
		}
		else
		{
			return num;
		}
	}

	/**
	 * If num is within tolerance of snapNum, snapNum is returned.
	 * Otherwise, num is returned.
	 */
	public static float snapTo( float num, float tolerance, float snapNum )
	{
		if ( Math.abs( snapNum - num ) < tolerance )
		{
			return snapNum;
		}
		else
		{
			return num;
		}
	}

	/**
	 * Interprets val as unsigned.  Must return value as the next bigger
	 * numerical type, since val's type can't represent the unsigned values.
	 */
	public static short asUnsigned( byte val )
	{
		return val >= 0 ? val : (short) ( 256 + (short) val );
	}

	/**
	 * Interprets val as unsigned.  Must return value as the next bigger
	 * numerical type, since val's type can't represent the unsigned values.
	 */
	public static int asUnsigned( short val )
	{
		return val >= 0 ? val : ( (int) 65536 + (int) val );
	}

	/**
	 * Interprets val as unsigned.  Must return value as the next bigger
	 * numerical type, since val's type can't represent the unsigned values.
	 */
	public static long asUnsigned( int val )
	{
		return val >= 0 ? val : ( 4294967296L + (long) val );
	}

	public static Random getRandom()
	{
		return unsyncedRandom_;
	}

	public static double mod( double val, double modval )
	{
		double retval;
		//retval + n*modval = val, 0 <= retval < modval
		//retval = val - n*modval
		//n = (retval - val)/modval
		double n = Math.floor( val / modval );
		retval = val - n * modval;
		return retval;
	}

	public static float mod( float val, float modval )
	{
		float retval;
		//retval + n*modval = val, 0 <= retval < modval
		//retval = val - n*modval
		//n = (retval - val)/modval
		float n = (float) Math.floor( val / modval );
		retval = val - n * modval;
		return retval;
	}

	/**
	 * The pctNum is a fraction written as a percent, i.e. pctNum = 25 means
	 * 25% or .25 or 1/4.  Returns pctNum percent of value.
	 */
	public static float pctOf( int pctNum, int value )
	{
		return ( (float) pctNum * value ) / (float) 100;
	}

	protected static Random unsyncedRandom_ = new Random();

	//com.navtools.networking.armi.networking.test code follows
	public static void main( String[] args )
	{
		testMod( 47.9, 2.4, 2.3 );
		testMod( -47.9, -2.4, -2.3 );
		testMod( -47.9, 2.4, .1 );
		testMod( -2.3, 2.4, .1 );
	}

	public static boolean testMod( double val, double modval, double realretval )
	{
		final double tolerance = 0.0001;

		double retval = mod( val, modval );
		if ( !equalsTolerance( retval, realretval, tolerance ) )
		{
			System.err.println( "Expected retval " + realretval +
			                    ", got retval " + retval );
			return false;
		}

		return true;
	}
}
