//Copyright:    Copyright (c) 2001
//Author:       Bobby Martin

package com.navtools.util;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

import com.navtools.thread.Lock;
import org.apache.log4j.Category;

//import com.sun.corba.se.internal.orbutil.Lock;
//import sun.misc.Lock;

public class MethodMatcher
{
	public static final Category LOG =
	Category.getInstance( MethodMatcher.class.getName() );

	/**
	 * Typically you'll just want to use the singleton instance that you
	 * access from instance()
	 */
	public MethodMatcher()
	{
	}

	public static MethodMatcher instance()
	{
		if ( instance_ == null )
		{
			instance_ = new MethodMatcher();
		}
		return instance_;
	}

	static public Class[] getClassesFromObjects( Object[] objList )
	{
		Class[] retval = new Class[objList.length];

		for ( int i = 0; i < objList.length; ++i )
		{
			retval[i] = ( objList[i] == null ) ? null : objList[i].getClass();
		}

		return retval;
	}

	/**
	 * Return true iff arg1 & arg2 have the same name, parameter list, and
	 * return value.
	 */
	public static boolean methodsMatch( Method arg1, Method arg2 )
	{
		return arg1.getName().equals( arg2.getName() ) && arg1.getReturnType() == arg2.getReturnType() &&
		       ArrayUtil.compare( arg1.getParameterTypes(), arg2.getParameterTypes() );
	}

	public Method getMethod( Object target,
	                         String simpleMethodName,
	                         Object[] args )
	{
		return getMethod( target.getClass(),
		                  simpleMethodName,
		                  getClassesFromObjects( args ) );
	}

	public Method getMethod( Class targetClass,
	                         String simpleMethodName,
	                         Class[] methodParmTypes )
	{
		Method method;
		Lock.logBeforeSync();
		synchronized ( this )
		{
			Lock.logBeginSync();
			String hashKey = makeHashKey( targetClass,
			                              simpleMethodName,
			                              methodParmTypes );
			method = (Method) cache_.get( hashKey );

			if ( method == null )
			{
				method = findMethod( targetClass, simpleMethodName, methodParmTypes );
				if ( method != null )
				{
					cache_.put( hashKey, method );
				}
			}
			Lock.logEndSync();
		}
		Lock.logAfterSync();

		return method;
	}

	//builds a string of the form Class.method(Arg1Class,Arg2Class,...) to be
	//the cache_ hashKey
	public static String toString( Class targetClass,
	                               String simpleMethodName,
	                               Class[] methodParmTypes )
	{
		StringBuffer buffer = new StringBuffer();
		buffer.append( targetClass.getName() + "." + simpleMethodName + "(" );

		boolean first = true;
		for ( int i = 0; i < methodParmTypes.length; ++i )
		{
			if ( !first )
			{
				buffer.append( "," );
			}
			first = false;

			buffer.append( ( methodParmTypes[i] == null )
			               ? "null" : methodParmTypes[i].getName() );
		}

		buffer.append( ")" );
		return buffer.toString();
	}

	//builds a string of the form Class.method(Arg1Class,Arg2Class,...) to be
	//the cache_ hashKey
	private String makeHashKey( Class targetClass,
	                            String simpleMethodName,
	                            Class[] methodParmTypes )
	{
		return toString( targetClass, simpleMethodName, methodParmTypes );
	}

	private Method findMethod( Class targetClass,
	                           String simpleMethodName,
	                           Class[] methodParmTypes )
	{
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "MultipleDispatchMethod finding method for signature " +
			           makeHashKey( targetClass, simpleMethodName, methodParmTypes ) );
		}
		Method[] methods = targetClass.getMethods();

		Method retval = null;
		int retvalMatchQuality = 0;
		int currMatchQuality;
		for ( int i = 0; i < methods.length; i++ )
		{
			if ( methods[i].getName().equals( simpleMethodName ) )
			{
				Class[] signature = methods[i].getParameterTypes();

				currMatchQuality = parmsMatchQuality( signature, methodParmTypes );
				if ( currMatchQuality > retvalMatchQuality )
				{
					retvalMatchQuality = currMatchQuality;
					retval = methods[i];
				}
			}
		}

		return retval;
	}

	static public int parmsMatchQuality( Class[] signature,
	                                     Class[] actualParms )
	{
		if ( signature.length != actualParms.length )
		{
			return -1;
		}

		int inheritanceSteps = 0;
		//sum up the number of superclasses that must be navigated to get from
		//the actual parm type to the signature type for all parameters
		for ( int i = 0; i < signature.length; ++i )
		{
			if ( actualParms[i] == null )
			{
				//a null parameter is not valid for a primitive argument,
				//so return the lowest quality
				if ( signature[i].isPrimitive() )
				{
					LOG.debug( "null value for primitive, returning quality -1" );
					return -1;
				}
				//a null parameter is not happy-making for an array argument,
				//so use lots of steps
				else
				{
					if ( signature[i].isArray() )
					{
						LOG.debug( "null value for array, using 200 steps" );
						inheritanceSteps += 100;
					}
					else
					{
						//we assume that the basemost class is better for
						//a null parameter, so we use the number of steps from
						//object to the signature type
						inheritanceSteps += calcStepsBetween( Object.class,
						                                      signature[i] );
					}
				}
			}
			else
			{
				if ( !canAssign( signature[i], actualParms[i] ) )
				{
					LOG.debug( "signature type is not assignable from actual " +
					           "param, returning -1" );
					return -1;
				}
				else
				{
					inheritanceSteps += calcStepsBetween( signature[i],
					                                      actualParms[i] );
				}
			}
		}

		return 100000 - inheritanceSteps;
	}

	public static int calcStepsBetween( Class parent, Class child )
	{
		int retval = calcStepsBetween( parent, child, 0 );
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Steps between " + parent.getName() + " and " + child.getName() +
			           " is " + retval );
		}
		return retval;
	}

	protected static int calcStepsBetween( Class parent, Class child, int steps )
	{
		if ( child == null )
		{
			return -1;
		}

		if ( child.equals( parent ) )
		{
			return steps;
		}
		if ( parent.isInterface() )
		{
			Class[] interfaces = child.getInterfaces();
			for ( int i = 0; i < interfaces.length; ++i )
			{
				if ( canAssign( parent, interfaces[i] ) )
				{
					return calcStepsBetween( parent, interfaces[i], steps + 1 );
				}
			}
		}
		else
		{
			return calcStepsBetween( parent, child.getSuperclass(), steps + 1 );
		}

		return steps;
	}

	public static boolean canAssign( Class to, Class from )
	{
		to = getNonPrimitive( to );
		from = getNonPrimitive( from );
		return to.isAssignableFrom( from );
	}

	protected static Map primitiveMap = new HashMap();

	static
	{
		primitiveMap.put( boolean.class, Boolean.class );
		primitiveMap.put( byte.class, Byte.class );
		primitiveMap.put( char.class, Character.class );
		primitiveMap.put( short.class, Short.class );
		primitiveMap.put( int.class, Integer.class );
		primitiveMap.put( long.class, Long.class );
		primitiveMap.put( float.class, Float.class );
		primitiveMap.put( double.class, Double.class );
	}

	public static Class getNonPrimitive( Class clazz )
	{
		Class retval = (Class) primitiveMap.get( clazz );
		return retval == null ? clazz : retval;
	}

	public Object classOrMethodForName( String classOrMethodName )
	{
		Object retval = null;

		//iff it doesn't have a ( in it, it's a class name
		if ( classOrMethodName.indexOf( "(" ) == -1 )
		{
			try
			{
				retval = Class.forName( classOrMethodName );
			}
			catch ( Exception e )
			{
				LOG.error( e.getMessage(), e );
			}
		}
		//otherwise it's a method name (if it has a ( in it)
		else
		{
			retval = MethodMatcher.instance().forName( classOrMethodName );
		}

		return retval;
	}

	public Method forName( String name )
	{
		//condition 'name'
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Unconditioned method name: " + name );
		}
		name = conditionMethodName( name );
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Conditioned method name: " + name );
		}

		Method retval;
		//if is in cache, return it now
		if ( ( retval = (Method) cache_.get( name ) ) != null )
		{
			return retval;
		}

		//parse for class name
		StringTokenizer toker = new StringTokenizer( name, " \t(" );

		//throw away return value
		toker.nextToken();
		String fullyQualifiedMethodName = toker.nextToken();

		int lastDot = fullyQualifiedMethodName.lastIndexOf( '.' );
		//String methodName = fullyQualifiedMethodName.substring(lastDot);
		String className = fullyQualifiedMethodName.substring( 0, lastDot );
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Class name: " + className );
		}

		try
		{
			//get all methods in class
			Class theClass = Class.forName( className );
			Method[] methods = theClass.getMethods();

			//for each method in class
			for ( int i = 0; i < methods.length; ++i )
			{
				//go ahead and put all methods in cache
				String currMethodName = conditionMethodName( methods[i].toString() );
				cache_.put( currMethodName, methods[i] );

				//get name of method
				//condition method name to match 'name' if possible
				//compare 'name' to method name, return if matches
				if ( name.equals( conditionMethodName( methods[i].toString() ) ) )
				{
					return methods[i];
				}
			}
		}
		catch ( ClassNotFoundException e )
		{
			LOG.error( e.getMessage(), e );
		}

		//if no matching method, return null
		return null;
	}

	public String conditionMethodName( String name )
	{
		String[] reservedWords = {"public", "private", "protected",
		                          "static", "synchronized", "native",
		                          "abstract", "final"};
		//TODO: verify format is acceptable
		//method format: "(((public)|(private)|(protected)|(static))\s+)*[a-zA-Z.]+\s+[a-zA-Z.]+\s*\((([a-zA-Z.]+\s*,)*[a-zA-Z.]+)?\)\s*(throws\s.*)?

		//throw away everything after 'throws'
		int throwsIndex = name.indexOf( "throws " );
		if ( throwsIndex != -1 )
		{
			name = name.substring( 0, throwsIndex - 1 );
		}

		//tokenize by whitespace
		StringTokenizer toker = new StringTokenizer( name, " \t" );
		StringBuffer retval = new StringBuffer();
		boolean first = true;

		while ( toker.hasMoreTokens() )
		{
			String token = toker.nextToken();
			//throw away reserved words
			if ( ArrayUtil.searchEquals( reservedWords, token ) !=
			     ArrayUtil.NOT_FOUND )
			{
				continue;
			}

			//put pieces back together with one space between each piece
			if ( !first )
			{
				retval.append( " " );
			}
			first = false;
			retval.append( token );
		}

		return retval.toString();
	}

	protected Map cache_ = new HashMap();

	protected static MethodMatcher instance_;

	//com.navtools.networking.armi.networking.test code follows
	public static void main( String[] args ) throws Exception
	{
		boolean verbose = false;

		String className = args[0];
		MethodMatcher mm = MethodMatcher.instance();

		//get all methods in class
		Class theClass = Class.forName( className );
		Method[] methods = theClass.getMethods();

		//for each method in class
		for ( int i = 0; i < methods.length; ++i )
		{
			//go ahead and put all methods in cache
			String currMethodName = mm.conditionMethodName( methods[i].toString() );
			if ( verbose )
			{
				System.out.println( currMethodName );
			}

			if ( !mm.classOrMethodForName( methods[i].toString() ).equals( methods[i] ) )
			{
				System.out.println();
				System.out.println( "Error looking up/caching method\n" +
				                    methods[i].toString() );
				System.out.println( "Method from lookup: " + mm.classOrMethodForName( methods[i].toString() ) );
				System.out.println( "Actual method: " + methods[i] );
				System.out.println();
			}
		}
	}
}
