package com.navtools.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.*;

import org.apache.log4j.Category;

/**
 * Class that monitors performance by gathering various data.
 * Not much in here right now...
 * Needs the file performance.prop to initialize in a good way. If this file is not found nothing will be monitored.
 * @author: Henrik Eriksson
 */
public class PerformanceMonitor
{

	public static final Category LOG =
	Category.getInstance( PerformanceMonitor.class.getName() );

	private static PerformanceMonitor instance;

	private Map throughputMap;
	private Map msgBuildTimeMap;
	private Map msgSendTimeMap;
	private Map mcReconstructTimeMap;

	private Thread snapshotThread;

	// throughput variables (same as for ThroughputPos)
	private long startTime;
	private long lastTime;
	private int bytesInLastSec;
	private int bytesOutLastSec;
	private int bytesInActive;
	private int bytesOutActive;
	private long bytesInTotal;
	private long bytesOutTotal;

	// settings
	private boolean checkForChanges;
	private int checkInterval;
	private int snapshotInterval;
	private boolean enabled;
	private boolean throughputEnabled;
	private boolean msgBuildTimeEnabled;
	private boolean msgSendTimeEnabled;
	private boolean mcReconstructTimeEnabled;
	private int output;
	private File propFile;
	private File outputFile;

	/**
	 * Each object should be mapped to a MessengerID.
	 */
	class PerformanceDataPos
	{
		private int count;
		private long data;

		public PerformanceDataPos( long data )
		{
			count = 1;
			this.data = data;
		}

		public void addData( long data )
		{
			count++;
			this.data += data;
		}

		public int getAverage()
		{
			return (int) data / count;
		}

		public long getTotal()
		{
			return data;
		}

	}


	class ThroughputPos
	{
		private long startTime;
		private long lastTime;
		private int bytesInLastSec;
		private int bytesOutLastSec;
		private int bytesInActive;
		private int bytesOutActive;
		private long bytesInTotal;
		private long bytesOutTotal;


		public ThroughputPos( long start, int bytes, boolean outgoing )
		{
			super();
			startTime = start;
			lastTime = start;
			if ( outgoing )
			{
				bytesOutActive = bytes;
				bytesOutTotal = bytes;
			}
			else
			{
				bytesInActive = bytes;
				bytesInTotal = bytes;
			}
		}

		public void addIncomingBytes( long time, int bytes )
		{
			if ( time > lastTime + 999 )
			{
				lastTime = time;
				bytesInLastSec = bytesInActive;
				bytesInActive = bytes;
			}
			else
			{
				bytesInActive += bytes;
			}
			bytesInTotal += bytes;
		}

		public void addOutgoingBytes( long time, int bytes )
		{
			if ( time > lastTime + 999 )
			{
				lastTime = time;
				bytesOutLastSec = bytesOutActive;
				bytesOutActive = bytes;
			}
			else
			{
				bytesOutActive += bytes;
			}
			bytesOutTotal += bytes;
		}

		public int getBytesLastSec()
		{
			return bytesInLastSec + bytesOutLastSec;
		}

		public int getBytesInLastSec()
		{
			return bytesInLastSec;
		}

		public int getBytesOutLastSec()
		{
			return bytesOutLastSec;
		}

		public long getBytesTotal()
		{
			return bytesInTotal + bytesOutTotal;
		}

		public long getBytesInTotal()
		{
			return bytesInTotal;
		}

		public long getBytesOutTotal()
		{
			return bytesOutTotal;
		}

		public int getAvgThroughputSec()
		{
			int timeInSecs = (int) Math.round( ( System.currentTimeMillis() - startTime ) / 1000 );
			return (int) ( bytesInTotal + bytesOutTotal ) / timeInSecs;
		}

		public int getAvgBytesInSec()
		{
			int timeInSecs = (int) Math.round( ( System.currentTimeMillis() - startTime ) / 1000 );
			return (int) bytesInTotal / timeInSecs;
		}

		public int getAvgBytesOutSec()
		{
			int timeInSecs = (int) Math.round( ( System.currentTimeMillis() - startTime ) / 1000 );
			return (int) bytesOutTotal / timeInSecs;
		}
	}


	private PerformanceMonitor()
	{
		super();
		initialize();
		if ( !enabled )
		{
			return;
		}
		throughputMap = Collections.synchronizedMap( new HashMap() );
		msgBuildTimeMap = Collections.synchronizedMap( new HashMap() );
		msgSendTimeMap = Collections.synchronizedMap( new HashMap() );
		mcReconstructTimeMap = Collections.synchronizedMap( new HashMap() );
	}

	public void addMsgBuildTime( Long id, long time )
	{
		if ( !msgBuildTimeEnabled )
		{
			return;
		}
		PerformanceDataPos pdp = (PerformanceDataPos) msgBuildTimeMap.get( id );
		if ( pdp == null )
		{
			msgBuildTimeMap.put( id, new PerformanceDataPos( time ) );
		}
		else
		{
			pdp.addData( time );
		}
	}

	public void addMsgSendTime( Long id, long time )
	{
		if ( !msgSendTimeEnabled )
		{
			return;
		}
		PerformanceDataPos pdp = (PerformanceDataPos) msgSendTimeMap.get( id );
		if ( pdp == null )
		{
			msgSendTimeMap.put( id, new PerformanceDataPos( time ) );
		}
		else
		{
			pdp.addData( time );
		}
	}

	public void addMcReconstructTime( Long id, long time )
	{
		if ( !mcReconstructTimeEnabled )
		{
			return;
		}
		PerformanceDataPos pdp = (PerformanceDataPos) mcReconstructTimeMap.get( id );
		if ( pdp == null )
		{
			mcReconstructTimeMap.put( id, new PerformanceDataPos( time ) );
		}
		else
		{
			pdp.addData( time );
		}
	}

	public int getAverageMsgBuildTime( Long id )
	{
		PerformanceDataPos pdp = (PerformanceDataPos) msgBuildTimeMap.get( id );
		if ( pdp == null )
		{
			return 0;
		}

		return pdp.getAverage();

	}

	public long getTotalMsgBuildTime( Long id )
	{
		PerformanceDataPos pdp = (PerformanceDataPos) msgBuildTimeMap.get( id );
		if ( pdp == null )
		{
			return 0;
		}

		return pdp.getTotal();
	}

	public int getAverageMsgSendTime( Long id )
	{
		PerformanceDataPos pdp = (PerformanceDataPos) msgSendTimeMap.get( id );
		if ( pdp == null )
		{
			return 0;
		}

		return pdp.getAverage();

	}

	public long getTotalMsgSendTime( Long id )
	{
		PerformanceDataPos pdp = (PerformanceDataPos) msgSendTimeMap.get( id );
		if ( pdp == null )
		{
			return 0;
		}

		return pdp.getTotal();
	}

	public int getAverageMcReconstructTime( Long id )
	{
		PerformanceDataPos pdp = (PerformanceDataPos) mcReconstructTimeMap.get( id );
		if ( pdp == null )
		{
			return 0;
		}

		return pdp.getAverage();

	}

	public long getTotalMcReconstructTime( Long id )
	{
		PerformanceDataPos pdp = (PerformanceDataPos) mcReconstructTimeMap.get( id );
		if ( pdp == null )
		{
			return 0;
		}

		return pdp.getTotal();
	}

	/**
	 * Add a number of incoming bytes to the monitor.
	 */
	public void addIncomingBytes( Long id, long time, int bytes )
	{
		if ( !throughputEnabled )
		{
			return;
		}
		if ( startTime == 0 )
		{
			startTime = time;
		}
		if ( time > lastTime + 999 )
		{
			lastTime = time;
			bytesInLastSec = bytesInActive;
			bytesInActive = bytes;
		}
		else
		{
			bytesInActive += bytes;
		}
		bytesInTotal += bytes;
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			throughputMap.put( id, new ThroughputPos( time, bytes, false ) );
		}
		else
		{
			tp.addIncomingBytes( time, bytes );
		}
	}

	/**
	 * Add a number of outgoing bytes to the monitor.
	 */
	public void addOutgoingBytes( Long id, long time, int bytes )
	{
		if ( !throughputEnabled )
		{
			return;
		}
		if ( startTime == 0 )
		{
			startTime = time;
		}
		if ( time > lastTime + 999 )
		{
			lastTime = time;
			bytesOutLastSec = bytesOutActive;
			bytesOutActive = bytes;
		}
		else
		{
			bytesOutActive += bytes;
		}
		bytesOutTotal += bytes;
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			throughputMap.put( id, new ThroughputPos( time, bytes, true ) );
		}
		else
		{
			tp.addOutgoingBytes( time, bytes );
		}
	}

	/**
	 * Gets the average number of incoming bytes per second for a certain MessengerID (id).
	 */
	public int getAverageIncomingBytesSec( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getAvgBytesInSec();
	}

	/**
	 * Gets the average number of incoming bytes per second (total).
	 */
	public int getAverageIncomingBytesSec()
	{
		int timeInSecs = (int) Math.round( ( System.currentTimeMillis() - startTime ) / 1000 );
		return (int) bytesInTotal / timeInSecs;
	}

	/**
	 * Gets the average number of outgoing bytes per second for a certain MessengerID (id).
	 */
	public int getAverageOutgoingBytesSec( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getAvgBytesOutSec();
	}

	/**
	 * Gets the average number of outgoing bytes per second (total).
	 */
	public int getAverageOutgoingBytesSec()
	{
		int timeInSecs = (int) Math.round( ( System.currentTimeMillis() - startTime ) / 1000 );
		return (int) bytesOutTotal / timeInSecs;
	}

	/**
	 * Gets the average throughput (incoming + outgoing) in bytes per second for a certain MessengerID (id).
	 */
	public int getAverageThroughputSec( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getAvgThroughputSec();
	}

	/**
	 * Gets the average throughput (incoming + outgoing) in bytes per second (total).
	 */
	public int getAverageThroughputSec()
	{
		int timeInSecs = (int) Math.round( ( System.currentTimeMillis() - startTime ) / 1000 );
		return (int) ( bytesInTotal + bytesOutTotal ) / timeInSecs;
	}

	/**
	 * Gets the number of incoming bytes last second for a certain MessengerID (id).
	 */
	public int getIncomingBytesLastSec( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getBytesInLastSec();
	}

	/**
	 * Gets the number of incoming bytes last second (total).
	 */
	public int getIncomingBytesLastSec()
	{
		return bytesInLastSec;
	}

	/**
	 * Gets the total number of incoming bytes for a certain MessengerID (id).
	 */
	public long getIncomingBytesTotal( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getBytesInTotal();
	}

	/**
	 * Gets the total number of incoming bytes (total).
	 */
	public long getIncomingBytesTotal()
	{
		return bytesInTotal;
	}

	/**
	 * Gets the number of outgoing bytes last second for a certain MessengerID (id).
	 */
	public int getOutgoingBytesLastSec( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getBytesOutLastSec();
	}

	/**
	 * Gets the number of outgoing bytes last second (total).
	 */
	public int getOutgoingBytesLastSec()
	{
		return bytesOutLastSec;
	}

	/**
	 * Gets the total number of outgoing bytes for a certain MessengerID (id).
	 */
	public long getOutgoingBytesTotal( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getBytesOutTotal();
	}

	/**
	 * Gets the total number of outgoing bytes (total).
	 */
	public long getOutgoingBytesTotal()
	{
		return bytesOutTotal;
	}

	/**
	 * Gets the total throughput (incoming + outgoing) in bytes last second for a certain MessengerID (id).
	 */
	public int getThroughputLastSec( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getBytesLastSec();
	}

	/**
	 * Gets the total throughput (incoming + outgoing) in bytes last second (total).
	 */
	public int getThroughputLastSec()
	{
		return bytesInLastSec + bytesOutLastSec;
	}

	/**
	 * Gets the total throughput (incoming + outgoing) in bytes for a certain MessengerID (id).
	 */
	public long getTotalThroughput( Long id )
	{
		ThroughputPos tp = (ThroughputPos) throughputMap.get( id );
		if ( tp == null )
		{
			return 0;
		}

		return tp.getBytesTotal();
	}

	/**
	 * Gets the total throughput (incoming + outgoing) in bytes (total).
	 */
	public long getTotalThroughput()
	{
		return bytesInTotal + bytesOutTotal;
	}

	public void initialize()
	{
		// look for property file to get settings from
		propFile = new File( "performance.prop" );
		final long lastModified = propFile.lastModified();
		if ( propFile.exists() )
		{
			Properties props = new Properties();
			try
			{
				props.load( new FileInputStream( propFile ) );
				// if watch_for_changes is true we start a watcher thread that checks the prop file for changes
				// every 20 seconds if nothing else is set in check_interval
				checkForChanges = Boolean.valueOf( props.getProperty( "watch_for_changes" ) ).booleanValue();
				if ( checkForChanges )
				{
					checkInterval = 20;
					try
					{
						checkInterval = Integer.parseInt( props.getProperty( "check_interval" ) ) * 1000;
					}
					catch ( NumberFormatException nfe )
					{
					}
					new Thread( "performance watcher" )
					{
						public void run()
						{
							long modified = lastModified;
							while ( true )
							{
								try
								{
									Thread.currentThread().sleep( checkInterval );
									long start = System.currentTimeMillis();
									File f = new File( "performance.prop" );
									Properties p = new Properties();

									p.load( new FileInputStream( f ) );
									if ( modified != f.lastModified() )
									{
										modified = f.lastModified();
										checkForChanges = Boolean.valueOf( p.getProperty( "watch_for_changes" ) ).booleanValue();
										if ( checkForChanges )
										{
											try
											{
												checkInterval = Integer.parseInt( p.getProperty( "check_interval" ) ) * 1000;
											}
											catch ( NumberFormatException nfe )
											{
											}
										}
										initialize( p );

										if ( !checkForChanges )
										{
											break;
										}
									}
									System.out.println( "time to init: " + ( System.currentTimeMillis() - start ) );
								}
								catch ( Exception e )
								{
									LOG.warn( "Exception in watcher thread: " + e.getMessage() );
								}

							}
						}
					}.start();
				}

				initialize( props );
			}
			catch ( IOException ioe )
			{
				LOG.warn( ioe.getMessage() );
			}
		}
		else
		{
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug(
				"The file performance.prop was not found. No Performance monitoring will be enabled" );
			}
		}

	}


	private void initialize( Properties props )
	{

		// initialize the settings from the properties
		enabled = Boolean.valueOf( props.getProperty( "enabled" ) ).booleanValue();
		if ( !enabled )
		{
			throughputEnabled = false;
			msgBuildTimeEnabled = false;
			msgSendTimeEnabled = false;
			mcReconstructTimeEnabled = false;
			snapshotInterval = 0;
			return;
		}
		throughputEnabled = Boolean.valueOf( props.getProperty( "throughput_enabled" ) ).booleanValue();
		msgBuildTimeEnabled = Boolean.valueOf( props.getProperty( "msg_build_time_enabled" ) ).booleanValue();
		msgSendTimeEnabled = Boolean.valueOf( props.getProperty( "msg_send_time_enabled" ) ).booleanValue();
		mcReconstructTimeEnabled = Boolean.valueOf( props.getProperty( "mc_reconstruct_time_enabled" ) ).booleanValue();
		snapshotInterval = getIntProp( props, "snapshot_interval" );
		// if snapshot interval is bigger than 0 and a snapshot thread isn't already running, start one...
		if ( snapshotInterval > 0 && !( snapshotThread != null && snapshotThread.isAlive() ) )
		{
			snapshotThread = new Thread( "snapshot" )
			{
				public void run()
				{
					try
					{
						while ( snapshotInterval > 0 )
						{
							Thread.currentThread().sleep( snapshotInterval * 1000 );
							snapshot();
						}
					}
					catch ( InterruptedException ie )
					{
						LOG.warn( "Snapshot thread was interrupted" );
					}
				}
			};
			snapshotThread.start();
		}

	}

	private int getIntProp( Properties p, String name )
	{
		int retVal = -1;
		try
		{
			retVal = Integer.parseInt( p.getProperty( name ) );
		}
		catch ( NumberFormatException ne )
		{
		}
		return retVal;
	}

	public static PerformanceMonitor instance()
	{
		if ( instance == null )
		{
			instance = new PerformanceMonitor();
		}
		return instance;
	}

	public boolean isEnabled()
	{
		return enabled;
	}

	public boolean isThroughputEnabled()
	{
		return throughputEnabled;
	}

	public boolean isMsgBuildTimeEnabled()
	{
		return msgBuildTimeEnabled;
	}

	public boolean isMsgSendTimeEnabled()
	{
		return msgSendTimeEnabled;
	}

	public boolean isMcReconstructTimeEnabled()
	{
		return mcReconstructTimeEnabled;
	}

	public void snapshot()
	{
		// check what output to use first - we're skipping that for now and dump everything to stdout
		if ( throughputEnabled )
		{
			Iterator it = throughputMap.keySet().iterator();
			System.out.println( "THROUGHPUT" );
			while ( it.hasNext() )
			{
				Long m = (Long) it.next();
				System.out.println( "Incoming bytes total for " + m + " " + getIncomingBytesTotal( m ) );
				System.out.println( "Incoming bytes last second for " + m + " " + getIncomingBytesLastSec( m ) );
				System.out.println( "Outgoing bytes total for " + m + " " + getOutgoingBytesTotal( m ) );
				System.out.println( "Outgoing bytes last second for " + m + " " + getOutgoingBytesLastSec( m ) );
				System.out.println( "Average incoming bytes per second for " + m + " " + getAverageIncomingBytesSec( m ) );
				System.out.println( "Average outgoing bytes per second for " + m + " " + getAverageOutgoingBytesSec( m ) );
				System.out.println( "Average throughput bytes per second for " + m + " " + getAverageThroughputSec( m ) );
				System.out.println( "Total hroughput in bytes for " + m + " " + getTotalThroughput( m ) );
				System.out.println( "Throughput last sec for " + m + " " + getThroughputLastSec( m ) );
				System.out.println( "----------" );
			}
			System.out.println( "Incoming bytes total " + getIncomingBytesTotal() );
			System.out.println( "Incoming bytes last second " + getIncomingBytesLastSec() );
			System.out.println( "Outgoing bytes total " + getOutgoingBytesTotal() );
			System.out.println( "Outgoing bytes last second " + getOutgoingBytesLastSec() );
			System.out.println( "Average incoming bytes per second " + getAverageIncomingBytesSec() );
			System.out.println( "Average outgoing bytes per second " + getAverageOutgoingBytesSec() );
			System.out.println( "Average throughput bytes per second " + getAverageThroughputSec() );
			System.out.println( "Total throughput in bytes " + getTotalThroughput() );
			System.out.println( "Throughput last sec " + getThroughputLastSec() );
			System.out.println( "----------" );
		}
		if ( msgBuildTimeEnabled )
		{
			Iterator it = msgBuildTimeMap.keySet().iterator();
			System.out.println( "MESSAGE BUILD TIME" );
			while ( it.hasNext() )
			{
				Long m = (Long) it.next();
				System.out.println( "Average message build time for " + m + " " + getAverageMsgBuildTime( m ) );
				System.out.println( "Total message build time for " + m + " " + getTotalMsgBuildTime( m ) );
				System.out.println( "----------" );
			}
		}
		if ( msgSendTimeEnabled )
		{
			Iterator it = msgSendTimeMap.keySet().iterator();
			System.out.println( "MESSAGE SEND TIME" );
			while ( it.hasNext() )
			{
				Long m = (Long) it.next();
				System.out.println( "Average message send time for " + m + " " + getAverageMsgSendTime( m ) );
				System.out.println( "Total message send time for " + m + " " + getTotalMsgSendTime( m ) );
				System.out.println( "----------" );
			}
		}
		if ( mcReconstructTimeEnabled )
		{
			Iterator it = mcReconstructTimeMap.keySet().iterator();
			System.out.println( "METHOD CALL RECONSTRUCTION TIME" );
			while ( it.hasNext() )
			{
				Long m = (Long) it.next();
				System.out.println( "Average method reconstruction time for " + m + " " + getAverageMcReconstructTime( m ) );
				System.out.println( "Total method reconstruction time for " + m + " " + getTotalMcReconstructTime( m ) );
				System.out.println( "----------" );
			}
		}
	}


}
