/*
 * $Id: SNTPClient.java,v 1.2 2003/09/06 21:49:22 wurp Exp $
 * $Log: SNTPClient.java,v $
 * Revision 1.2  2003/09/06 21:49:22  wurp
 * Migrated stuff from ARMI into here
 *
 * Revision 1.1  2003/08/18 17:12:42  gergiskhan
 * Refactoring.  Changed package structure.
 *
 * Revision 1.1  2003/08/08 02:17:21  gergiskhan
 * no message
 *
 * Revision 1.1  2002/04/15 03:43:57  wurp
 * Cache id to method mappings in text file
 * Fix sporadic bug in acquiring ids (timing bug in ClassAndMethodTable)
 * Fixed bug with methods having no arguments using ReturnValues
 * Added disconnectionListener
 *
 * Revision 1.4  2001/06/23 16:24:00  wurp
 * More log4j migration
 * Fixed sr; replacement of $1 type variables was broken.
 *
 * Revision 1.3  2001/06/22 03:06:02  wurp
 * More log4j changes
 *
 * Revision 1.2  2001/06/20 02:58:52  wurp
 * Updated to log4j.  Added default ctor to CharacterInfo
 *
 * Revision 1.1  2001/04/04 15:46:35  wurp
 * Added SNTP client to sync time from client to server.
 * Still have to run SNTP server on LoginServer, and may need to change from default SNTP port. (since it's below 1024 and thus not accessible to non-root processes on *nix)
 *
 */

package com.navtools.util;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.util.Random;

import org.apache.log4j.Category;

public class SNTPClient extends Thread
{
	public static final Category LOG =
	Category.getInstance( SNTPClient.class.getName() );

	protected SNTPClient()
	{
	}

	public static void initialize( TimeProvider timeProvider,
	                               InetAddress server,
	                               int port ) throws SocketException
	{
		timeProvider_ = timeProvider;
		timeServerAddress_ = server;
		port_ = port;
		socket_ = new DatagramSocket();
		new SNTPClient().start();
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Starting SNTPClient" );
		}
	}

	public static void initialize( TimeProvider timeProvider,
	                               InetAddress server ) throws SocketException
	{
		initialize( timeProvider, server, SNTPServer.DEFAULT_NTP_PORT );
	}

	public static void initialize( InetAddress server ) throws SocketException
	{
		initialize( defaultTimeProvider(), server, SNTPServer.DEFAULT_NTP_PORT );
	}

	public static void initialize( InetAddress server,
	                               int port ) throws SocketException
	{
		initialize( defaultTimeProvider(), server, port );
	}

	public static TimeProvider defaultTimeProvider()
	{
		if ( defaultTimeProvider_ == null )
		{
			defaultTimeProvider_ = new TimeProvider()
			{
				public long currentTimeMillis()
				{
					return System.currentTimeMillis();
				}
			};
		}

		return defaultTimeProvider_;
	}

	public static int getOffsetInMillis( int numTimesToPoll ) throws IOException
	{
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Getting offset" );
		}
		synchronized ( SNTPClient.class )
		{
			numTimesToPoll_ = Math.min( numTimesToPoll, MAX_NUM_POLLS );
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "numTimesToPoll: " + numTimesToPoll_ );
			}
			currNumPackets_ = 0;

			sendSNTPRequest();

			while ( currNumPackets_ < numTimesToPoll_ )
			{
				try
				{
					SNTPClient.class.wait();
				}
				catch ( Exception e )
				{
					LOG.error( e.getMessage(), e );
				}
			}

			return offset_;
		}
	}

	public static final void sendSNTPRequest() throws IOException
	{
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Sending SNTP request" );
		}
		//initialize buffer to 0
		//set VN to 04; MODE to 03
		SNTPMessage msg = new SNTPMessage();

		nextExpectedKeyID_ = random_.nextInt();
		msg.setKeyIdentifier( nextExpectedKeyID_ );

		//preinitialize as much as possible to minimize time between
		//setting timestamp and sending packet
		DatagramPacket outgoingPacket_ =
		new DatagramPacket( msg.getBuffer(), msg.getBuffer().length,
		                    timeServerAddress_, port_ );

		//set Originate Timestamp to current time
		msg.setOriginateTimeStampFromMillis( timeProvider_.currentTimeMillis() );
		socket_.send( outgoingPacket_ );
	}

	public static final void receiveSNTPPacket( byte[] data, int length )
	throws IOException
	{
		//record destination time
		long destinationTimeInMillis = timeProvider_.currentTimeMillis();

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "Receiving packet of length " + length );
		}

		//if invalid packet, throw away
		SNTPMessage incoming = new SNTPMessage( data, length );
		if ( !incoming.isValid() ||
		     incoming.getKeyIdentifier() != nextExpectedKeyID_ )
		{
			LOG.error( "Invalid incoming message!" );
			//!!multiple returns
			return;
		}

		synchronized ( SNTPClient.class )
		{
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "in sync block" );
			}
			//calculate offset ((T2 - T1) + (T3 - T4)) / 2
			float offset = incoming.getOffsetInMillis( destinationTimeInMillis );
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "destination time in millis: " + destinationTimeInMillis );
			}

			//put offset in list
			offsets_[currNumPackets_] = offset;
			++currNumPackets_;

			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "Currently " + currNumPackets_ + " valid packets" );
			}
			//if num packets >= numTimesToPoll, process packets
			if ( currNumPackets_ >= numTimesToPoll_ )
			{
				processPackets();
			}
			//else request new packet
			else
			{
				sendSNTPRequest();
			}
		}
	}

	public static final void processPackets() throws IOException
	{
		//calculate average offset
		float mean = calcMean();

		//calculate dispersion in offset (a measure of the error in the
		//offsets)
		float dispersion = calcDispersion( mean );

		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( "mean is " + mean + ", dispersion is " + dispersion );
		}

		//throw away packets with variance^2 > 3*dispersion^2
		int j = 0;
		float variance;
		for ( int i = 0; i < offsets_.length; ++i )
		{
			variance = offsets_[i] - mean;
			if ( variance * variance <= 3 * dispersion * dispersion )
			{
				offsets_[j] = offsets_[i];
				++j;
			}
		}

		//get more packets until all packets have variance^2 <= 3*dispersion^2
		if ( j < numTimesToPoll_ )
		{
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "packets with unacceptable variance found, setting " +
				           "currNumPackets to " + j );
			}
			currNumPackets_ = j;
			sendSNTPRequest();
		}
		else  //if have received all packets needed
		{
			synchronized ( SNTPClient.class )
			{
				offset_ = Math.round( mean );
				SNTPClient.class.notifyAll();
			}
		}
	}

	public static final float calcMean()
	{
		float mean = 0;
		//average the offsets
		for ( int i = 0; i < currNumPackets_; ++i )
		{
			mean += offsets_[i];
		}
		mean /= currNumPackets_;

		return mean;
	}

	public static final float calcDispersion( float mean )
	{
		float dispersion = 0;

		//sum of square of diff between mean and each offset
		float variance;
		for ( int i = 0; i < currNumPackets_; ++i )
		{
			variance = offsets_[i] - mean;
			dispersion += variance * variance;
		}
		dispersion = (float) Math.sqrt( dispersion / currNumPackets_ );

		return dispersion;
	}

	public void run()
	{
		DatagramPacket incoming = new DatagramPacket( new byte[2048], 2048 );
		try
		{
			socket_.setSoTimeout( 10000 );
		}
		catch ( SocketException e )
		{
			LOG.error( "Couldn't set SO timeout for SNTPClient", e );
		}

		//while(true)
		while ( true )
		{
			try
			{
				try
				{
					incoming.setLength( 2048 );

					//receive packets on port ??
					socket_.receive( incoming );
					receiveSNTPPacket( incoming.getData(),
					                   incoming.getLength() );
				}
				catch ( InterruptedIOException e )
				{
					if ( currNumPackets_ < numTimesToPoll_ )
					{
						sendSNTPRequest();
					}
				}
			}
			catch ( Exception e )
			{
				LOG.error( e.getMessage(), e );
			}
		}
	}

	protected static final int MAX_NUM_POLLS = 50;

	protected static byte[] defaultOutgoingSNTPPacket_;
	protected static int numTimesToPoll_ = -1;
	protected static int currNumPackets_;
	protected static float[] offsets_ = new float[MAX_NUM_POLLS];
	protected static int offset_;
	protected static TimeProvider timeProvider_;
	protected static InetAddress timeServerAddress_;
	protected static int port_;
	protected static DatagramSocket socket_;
	protected static TimeProvider defaultTimeProvider_;
	protected static Random random_ = new Random();
	protected static int nextExpectedKeyID_;

	//debug code follows

	public static void main( String[] args ) throws Exception
	{
		System.out.println( "Starting SNTPClient" );
		SNTPClient.initialize( InetAddress.getByName( args[0] ) );
		System.out.println( "Started SNTPClient" );

		int[] testPolls = new int[]{1, 10, 20, 50};

		while ( true )
		{
			for ( int i = 0; i < testPolls.length; ++i )
			{
				System.out.println( "Offset in milliseconds for " +
				                    testPolls[i] + " polls: " +
				                    getOffsetInMillis( testPolls[i] ) );
			}

			try
			{
				Thread.currentThread().sleep( 5000 );
			}
			catch ( Exception e )
			{
				e.printStackTrace();
			}
		}
	}
}

//LI----------------------2  (00)
//VN----------------------3  (04)
//MODE--------------------3  (03 client, 04 server)
//STRATUM-----------------8  (00 client, 01 server)
//POLL--------------------8  (06; max interval between msgs of 2^6 sec)
//PRECISION---------------8  (-6; precision of local clock is 2^-6 sec)
//ROOT DELAY-------------32  (0; no root delay to primary ref)
//ROOT DISPERSION--------32  (0; no dispersion to primary ref)
//REFERENCE IDENTIFIER---32  (0 client, LOCL server)
//REFERENCE TIMESTAMP----64  (00; ignored)
//ORIGINATE TIMESTAMP----64  (client time when msg sent to svr)
//RECEIVE TIMESTAMP------64  (svr time when msg arrived at svr)
//TRANSMIT TIMESTAMP-----64  (svr time when msg sent to client)
//KEY IDENTIFIER---------32  (authentication, unused)
//MESSAGE DIGEST--------128  (authentication, unused)


/*
When the server reply is received, the client determines a Destination Timestamp variable as the time of arrival according to its clock in NTP timestamp format. The following table summarizes the four timestamps.

      Timestamp Name          ID   When Generated
      ------------------------------------------------------------
      Originate Timestamp     T1   time request sent by client
      Receive Timestamp       T2   time request received by server
      Transmit Timestamp      T3   time reply sent by server
      Destination Timestamp   T4   time reply received by client
The roundtrip delay d and local clock offset t are defined as

      d = (T4 - T1) - (T2 - T3)     t = ((T2 - T1) + (T3 - T4)) / 2.

T4 - .5*RT - T3
RT = (T4 - T1 - (T3 - T2))/2
(2*T4 - 2*T3 - (T4 - T1 - T3 + T2))/2
(T4 + T1 - T3 - T2)/2
((T1 - T2) + (T4 - T3))/2

*/
