/*
 * $Id: SNTPMessage.java,v 1.2 2003/09/06 21:49:23 wurp Exp $
 * $Log: SNTPMessage.java,v $
 * Revision 1.2  2003/09/06 21:49:23  wurp
 * Migrated stuff from ARMI into here
 *
 * Revision 1.1  2003/08/18 17:12:42  gergiskhan
 * Refactoring.  Changed package structure.
 *
 * Revision 1.1  2003/08/08 02:17:21  gergiskhan
 * no message
 *
 * Revision 1.1  2002/04/15 03:43:57  wurp
 * Cache id to method mappings in text file
 * Fix sporadic bug in acquiring ids (timing bug in ClassAndMethodTable)
 * Fixed bug with methods having no arguments using ReturnValues
 * Added disconnectionListener
 *
 * Revision 1.3  2001/06/22 03:06:02  wurp
 * More log4j changes
 *
 * Revision 1.2  2001/06/20 02:58:53  wurp
 * Updated to log4j.  Added default ctor to CharacterInfo
 *
 * Revision 1.1  2001/04/04 15:46:35  wurp
 * Added SNTP client to sync time from client to server.
 * Still have to run SNTP server on LoginServer, and may need to change from default SNTP port. (since it's below 1024 and thus not accessible to non-root processes on *nix)
 *
 */

package com.navtools.util;

import org.apache.log4j.Category;

public class SNTPMessage
{
	public static final Category LOG =
	Category.getInstance( SNTPMessage.class.getName() );

	public static final byte MODE_CLIENT = 3;
	public static final byte MODE_SERVER = 4;
	public static final int LOCAL = 0x4c4f434c;    //'LOCL'
	public static final int MIN_BUFF_LENGTH = 52;
	public static final int ORIGINATE_TIME_OFFSET = 24;
	public static final int RECEIVE_TIME_OFFSET = 32;
	public static final int TRANSMIT_TIME_OFFSET = 40;
	public static final int KEY_IDENTIFIER_OFFSET = 48;
	public static final double TWO_TO_THE_32 = (double) Math.pow( 2, 32 );

	//I don't use the message digest
	protected byte[] buffer_ = new byte[MIN_BUFF_LENGTH];

	protected boolean valid_ = true;

	/**
	 * Constructs a default SNTP message for origination from client
	 */
	public SNTPMessage()
	{
		setVersion( 4 );
		setMode( MODE_CLIENT );
	}

	/**
	 * Constructs  SNTP message from the buffer of an incoming SNTP
	 * message
	 */
	public SNTPMessage( byte[] buf )
	{
		this( buf, buf.length );
	}

	/**
	 * Constructs  SNTP message from the buffer of an incoming SNTP
	 * message
	 */
	public SNTPMessage( byte[] buf, int length )
	{
		if ( length >= MIN_BUFF_LENGTH )
		{
			System.arraycopy( buf, 0, buffer_, 0, buffer_.length );
		}
		else
		{
			valid_ = false;
		}
	}

	public byte[] getBuffer()
	{
		return buffer_;
	}

	public boolean isValid()
	{
		return valid_;
	}

	public void setServerData()
	{
		setMode( MODE_SERVER );
		setStratum( 1 );
		setPrecision( -6 );
		setReferenceIdentifier( LOCAL );
	}

	public void setMode( int mode )
	{
		//TODO, but not important now
		//buffer_[0] &= 0x1f;
	}

	public void setPrecision( int precision )
	{
		//TODO, but not important now
	}

	public void setVersion( int version )
	{
		//TODO, but not important now
	}

	public void setStratum( int stratum )
	{
		//TODO, but not important now
	}

	public void setReferenceIdentifier( int referenceIdentifier )
	{
		//TODO, but not important now
	}

	/**
	 * Computes the time difference between the server and the client.
	 * @param destinationTime the time this packet was received at client
	 */
	public float getOffsetInMillis( long destinationTimeInMillis )
	{
		float offset = (float)
		( ( ( getReceiveTimeInMillis() - getOriginateTimeInMillis() ) +
		    ( getTransmitTimeInMillis() - destinationTimeInMillis ) ) / 2.0 );
		return offset;
	}

	public long getOriginateTimeInMillis()
	{
		return timeStampToMillis( getTimeStamp( buffer_, ORIGINATE_TIME_OFFSET ) );
	}

	public long getReceiveTimeInMillis()
	{
		return timeStampToMillis( getTimeStamp( buffer_, RECEIVE_TIME_OFFSET ) );
	}

	public long getTransmitTimeInMillis()
	{
		return timeStampToMillis( getTimeStamp( buffer_, TRANSMIT_TIME_OFFSET ) );
	}

	public final void setOriginateTimeStampFromMillis( long millis )
	{
		setTimeStamp( buffer_, ORIGINATE_TIME_OFFSET,
		              millisToTimeStamp( millis ) );
	}

	public final void setReceiveTimeStampFromMillis( long millis )
	{
		setTimeStamp( buffer_, RECEIVE_TIME_OFFSET, millisToTimeStamp( millis ) );
	}

	public final void setTransmitTimeStampFromMillis( long millis )
	{
		setTimeStamp( buffer_, TRANSMIT_TIME_OFFSET, millisToTimeStamp( millis ) );
	}

	public final void setKeyIdentifier( int keyID )
	{
		RFCUtil.setInt( buffer_, KEY_IDENTIFIER_OFFSET, keyID );
	}

	public final int getKeyIdentifier()
	{
		return RFCUtil.getInt( buffer_, KEY_IDENTIFIER_OFFSET );
	}

	public static final long getTimeStamp( byte[] buf, int offset )
	{
		return RFCUtil.getLong( buf, offset );
	}

	public static final void setTimeStamp( byte[] buf, int offset,
	                                       long timeStamp )
	{
		RFCUtil.setLong( buf, offset, timeStamp );
	}

	public static final long timeStampToMillis( long timeStamp )
	{
		long retval = (long) ( ( ( (double) timeStamp ) / TWO_TO_THE_32 ) * 1000.0d );
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( timeStamp + " to millis" + retval );
		}
		return retval;
	}

	public static final long millisToTimeStamp( long millis )
	{
		long retval = (long) ( ( ( (double) millis ) / 1000.0d ) * TWO_TO_THE_32 );
		if ( LOG.isDebugEnabled() )
		{
			LOG.debug( millis + " to timestamp" + retval );
		}
		return retval;
	}


	/*
	private void intToBytes(byte [] buf, int offset, int i)
	{
	buf[offset] = (byte)((i >>> 24) & 0xff);
	buf[offset+1] = (byte)((i >>> 16) & 0xff);
	buf[offset+2] = (byte)((i >>> 8) & 0xff);
	buf[offset+3] = (byte)(i & 0xff);
	}

	private int bytesToInt(byte [] buf, int offset)
	{
	int r = 0;

	for (int i=0; i<4; i++)
	{
	  r <<= 8;
	  r |= ((int)(buf[offset+i]) & 0xff);
	}

	return r;
	}

	/** Converts this SNTPMsg object into a byte array
	  as specified in RFC1769.
	public byte[] toBytes()
	{
	byte buf[];
	int stat;

	buf = new byte[48];
	stat = 0;
	stat |= ((LeapIndicator & 0x3) << 30);
	stat |= ((Version & 0x7) << 27);
	stat |= ((Mode & 0x7) << 24);
	stat |= ((Stratum & 0xff) << 16);
	stat |= ((Poll & 0xff) << 8);
	stat |= (Precision & 0xff);

	intToBytes(buf, 0, stat);
	intToBytes(buf, 4, RootDelay);
	intToBytes(buf, 8, RootDispersion);
	intToBytes(buf, 12, ReferenceID);

	ReferenceTime.toBytes(buf, 16);
	OriginateTime.toBytes(buf, 24);
	ReceiveTime.toBytes(buf, 32);
	TransmitTime.toBytes(buf, 40);

	return buf;
	}

	/** Sets all the data members of this object based on
	  the suppled byte array in the manner specified in RFC1769.
	private void fromBytes(byte[] buf)
	{
	int stat;

	stat = bytesToInt(buf, 0);
	LeapIndicator = (byte) ((stat >>> 30) & 0x3);
	Version = (byte) ((stat >>> 27) & 0x7);
	Mode = (byte) ((stat >>> 24) & 0x7);
	Stratum = (byte) ((stat >>> 16) & 0xff);
	Poll = (byte) ((stat >>> 8) & 0xff);
	Precision = (byte) (stat & 0xff);

	RootDelay = bytesToInt(buf, 4);
	RootDispersion = bytesToInt(buf, 8);
	ReferenceID = bytesToInt(buf, 12);

	ReferenceTime = new NTPTimeStamp(buf, 16);
	OriginateTime = new NTPTimeStamp(buf, 24);
	ReceiveTime = new NTPTimeStamp(buf, 32);
	TransmitTime = new NTPTimeStamp(buf, 32);
	}

	public String toString()
	{
	StringBuffer sb = new StringBuffer();
	sb.append("LeapIndicator = "+LeapIndicator+"\n");
	sb.append("Version = "+Version+"\n");
	sb.append("Mode = "+Mode+"\n");
	sb.append("Stratum = "+Stratum+"\n");
	sb.append("Poll = "+Poll+"\n");
	sb.append("Precision = "+Precision+"\n");
	sb.append("RootDelay = "+RootDelay+"\n");
	sb.append("RootDispersion = "+RootDispersion+"\n");
	sb.append("ReferenceID = "+ReferenceID+"\n");
	sb.append("ReferenceTime = "+ReferenceTime+"\n");
	sb.append("OriginateTime = "+OriginateTime+"\n");
	sb.append("ReceiveTime   = "+ReceiveTime+"\n");
	sb.append("TransmitTime  = "+TransmitTime+"\n");
	return sb.toString();
	}
	public byte getLeapIndicator() {return LeapIndicator;}
	public byte getVersion() {return Version;}
	public byte getMode() {return Mode;}
	public byte getStratum() {return Stratum;}
	public byte getPoll() {return Poll;}
	public byte getPreciesion() {return Precision;}
	public int getRootDelay() {return RootDelay;}
	public int getRootDispersion() {return RootDispersion;}
	public int getReferenceID() {return ReferenceID;}
	public NTPTimeStamp getReferenceTime(){return ReferenceTime;}
	public NTPTimeStamp getOriginateTime(){return OriginateTime;}
	public NTPTimeStamp getReceiveTime(){return ReceiveTime;}
	public NTPTimeStamp getTransmitTime(){return TransmitTime;}

	public void setLeapIndicator(byte b) {LeapIndicator = (byte)(b & 0x3);}
	public void setVersion(byte b) {Version = (byte)(b & 0x7);}
	public void setMode(byte b) {Mode = (byte)(b & 0x7);}
	public void setStratum(byte b) {Stratum = b;}
	public void setPoll(byte b) {Poll = b;}
	public void setPrecision(byte b) {Precision = b;}
	public void setRootDelay(int i) {RootDelay = i;}
	public void setRootDispersion(int i) {RootDispersion = i;}
	public void setReferenceID(int i) {ReferenceID = i;}
	public void setReferenceTime(NTPTimeStamp ts){ReferenceTime = ts;}
	public void setOrginateTime(NTPTimeStamp ts){OriginateTime = ts;}
	public void setReceiveTime(NTPTimeStamp ts){ReceiveTime = ts;}
	public void setTransmitTime(NTPTimeStamp ts){TransmitTime = ts;}
	*/

	//debug code follows

	public static void main( String[] args )
	{
		long millis = System.currentTimeMillis();
		long timestamp = millisToTimeStamp( millis );
		System.out.println( millis + " equals timestamp " + timestamp );

		millis += 1000;
		timestamp = millisToTimeStamp( millis );
		System.out.println( millis + " equals timestamp " + timestamp );

		timestamp -= (long) ( Math.pow( 2, 32 ) );
		millis = timeStampToMillis( timestamp );
		System.out.println( timestamp + " equals millis " + millis );
	}
}
