/*
 * $Id: Timer.java,v 1.2 2003/09/06 21:49:23 wurp Exp $
 */

package com.navtools.util;

import java.util.*;

import com.navtools.thread.Lock;
import org.apache.log4j.Category;

public class Timer implements Runnable
{
	public static final Category LOG =
	Category.getInstance( Timer.class.getName() );

	public static class RemoveToken
	{
		public RemoveToken( TimedEvent e )
		{
			e_ = e;
		}


		protected TimedEvent e_;
	}


	public Timer()
	{
		new Thread( this ).start();
	}


	/**
	 * Add a timed event, where the first event occurs at now + e.period
	 * from epoch (or in other words, e.period from now).
	 */
	public RemoveToken add( TimedEvent e )
	{
		if ( e.getPeriodInMillis() < 0 ||
		     ( e.getPeriodInMillis() < 1 && e.isRepeating() ) )
		{
			throw new IllegalArgumentException( "Invalid period on timed event" );
		}

		long addTime = e.getPeriodInMillis() + getTimeInMillis();
		addInMillis( addTime, e );

		return new RemoveToken( e );
	}


	/**
	 * Add a timed event, where the first event occurs at timeInSeconds from
	 * epoch.
	 */
	public RemoveToken add( long timeInSeconds, TimedEvent e )
	{
		return addInMillis( timeInSeconds * 1000, e );
	}


	public RemoveToken addInMillis( long timeInMillis,
	                                TimedEvent e )
	{
		RemoveToken retval;

		Lock.logBeforeSync();
		synchronized ( this )
		{
			Lock.logBeginSync();

			Long tim = new Long( timeInMillis );
			List eventList = (List) timedEvents_.get( tim );
			if ( eventList == null )
			{
				eventList = new ArrayList();
				timedEvents_.put( tim, eventList );
			}

			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "", new Exception( "Adding event for " + tim ) );
			}

			e.setNextScheduledTime( timeInMillis );
			eventList.add( e );

			retval = new RemoveToken( e );

			Lock.logEndSync();
		}
		Lock.logAfterSync();

		return retval;
	}


	//the remove may be done during processing of an event, so it
	//can't change the list of events.  Instead, just add it to a list
	//that is guaranteed to be removed before the event would be next
	//processed.
	public void remove( RemoveToken toke )
	{
		if ( toke == null )
		{
			return;
		}

		Lock.logBeforeSync();
		synchronized ( pendingRemoves_ )
		{
			Lock.logBeginSync();
			pendingRemoves_.add( toke );
			Lock.logEndSync();
		}
		Lock.logAfterSync();
	}


	protected void doPendingRemoves()
	{
		Lock.logBeforeSync();
		synchronized ( pendingRemoves_ )
		{
			Lock.logBeginSync();
			Iterator iter = pendingRemoves_.iterator();
			while ( iter.hasNext() )
			{
				reallyRemove( (RemoveToken) iter.next() );
			}

			pendingRemoves_.clear();
			Lock.logEndSync();
		}
		Lock.logAfterSync();
	}


	protected synchronized void reallyRemove( RemoveToken toke )
	{
		Lock.logBeforeSync();
		synchronized ( this )
		{
			Lock.logBeginSync();
			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "Remove: seeking event for " + toke.e_.nextScheduledTime_ );
			}
			List eventList =
			(List) timedEvents_.get( new Long( toke.e_.nextScheduledTime_ ) );

			if ( eventList != null )
			{
				if ( LOG.isDebugEnabled() )
				{
					LOG.debug( "Remove: found event for " + toke.e_.nextScheduledTime_ );
				}
				eventList.remove( toke.e_ );
			}
			Lock.logEndSync();
		}
		Lock.logAfterSync();
	}


	public final long getTimeInSeconds()
	{
		return getTimeInMillis() / getSecondsMultiplier();
	}


	public long getTimeInMillis()
	{
		return System.currentTimeMillis();
	}


	public long getSecondsMultiplier()
	{
		return 1000;
	}


	public Date getDate()
	{
		return new Date( getTimeInSeconds() );
	}


	public static Timer instance()
	{
		if ( instance_ == null )
		{
			instance_ = new Timer();
		}

		return instance_;
	}


	public static void setInstance( Timer t )
	{
		instance_ = t;
	}


	public static void establishInstance( Timer t )
	{
		if ( instance_ != null )
		{
			throw new RuntimeException( "Timer instance already set to " +
			                            t.getClass().getName() );
		}

		setInstance( t );
	}


	public void run()
	{
		while ( true )
		{
			try
			{
				Thread.sleep( pollingPeriod_ );
			}
			catch ( InterruptedException e )
			{
				//don't care if interrupted
			}

//              System.out.println("Current time is: " + getTime());
			tock();
		}
	}


	public void tock()
	{
		try
		{
			Long nextTimeToProcess;
			while ( timedEvents_.entrySet().size() > 0 &&
			        ( nextTimeToProcess = getNextTimeToProcess() ).longValue() <=
			        getTimeInMillis() )
			{
				if ( LOG.isDebugEnabled() )
				{
					LOG.debug( "Processing next time to process: " +
					           nextTimeToProcess );
				}

				doPendingRemoves();

				if ( LOG.isDebugEnabled() )
				{
					LOG.debug( "Line 177" );
				}

				Map.Entry entry = getNextTimedEvent();
				List eventList = (List) entry.getValue();
				Iterator iter = eventList.iterator();
				while ( iter.hasNext() )
				{
					TimedEvent e = (TimedEvent) iter.next();
					//maybe use different thread to trigger event?
					try
					{
						e.trigger();
					}
					catch ( Exception exc )
					{
						LOG.error( "Uncaught exception in " +
						           "TimedEvent.trigger()", exc );
					}

					if ( e.isRepeating() )
					{
						addInMillis( nextTimeToProcess.longValue() + e.getPeriodInMillis(), e );
					}
				}
				timedEvents_.remove( nextTimeToProcess );
				doPendingRemoves();

				if ( LOG.isDebugEnabled() )
				{
					LOG.debug( timedEvents_.size() + " events in list." );
				}
			}
		}
		catch ( Exception ex )
		{
			LOG.error( ex.getMessage(), ex );
		}

//          System.out.println("Nothing processed.  Next time to process: " + nextTimeToProcess);
	}


	public Long getNextTimeToProcess()
	{
		return (Long) getNextTimedEvent().getKey();
	}


	public Map.Entry getNextTimedEvent()
	{
		Map.Entry retval;
		Lock.logBeforeSync();
		synchronized ( this )
		{
			Lock.logBeginSync();

			if ( LOG.isDebugEnabled() )
			{
				LOG.debug( "1 Entryset size: " +
				           timedEvents_.entrySet().size() );
			}

			retval = (Map.Entry) timedEvents_.entrySet().iterator().next();

			Lock.logEndSync();
		}
		Lock.logAfterSync();
		return retval;
	}


	public void setPollingPeriodInMillis( int millis )
	{
		pollingPeriod_ = millis;
	}


	//polling time in millis
	protected int pollingPeriod_ = 1000;
	protected Map timedEvents_ = Collections.synchronizedMap( new TreeMap() );
	protected List pendingRemoves_ = new ArrayList();

	protected static Timer instance_;

	//com.navtools.networking.armi.networking.test code follows

	//testing function
	public static void main( String[] args )
	{
		System.out.println( "You should see tick or tock alternating" );
		System.out.println( "every five seconds.  At 30 seconds you should" );
		System.out.println( "see a ding.  After 45 seconds, tick should" );
		System.out.println( "stop appearing, and you will only see tock" );
		System.out.println( "every 10 seconds." );

		final Timer timer = Timer.instance();
		timer.add( new TimedEvent( 30, false,
		                           new VoidFunction()
		                           {
			                           public Object execute()
			                           {
				                           System.out.println( "One time 30 second ding" );
				                           return null;
			                           }
		                           } ) );

		final RemoveToken tickToke = timer.add( new TimedEvent( 10, true,
		                                                        new VoidFunction()
		                                                        {
			                                                        public Object execute()
			                                                        {
				                                                        System.out.println( "tick" );
				                                                        return null;
			                                                        }
		                                                        } ) );


		timer.add( timer.getTimeInSeconds() + 5, new TimedEvent( 10, true,
		                                                         new VoidFunction()
		                                                         {
			                                                         public Object execute()
			                                                         {
				                                                         System.out.println( "tock" );
				                                                         return null;
			                                                         }
		                                                         } ) );

		timer.add( new TimedEvent( 45, false,
		                           new VoidFunction()
		                           {
			                           public Object execute()
			                           {
				                           System.out.println( "Removing tick" );
				                           timer.remove( tickToke );
				                           return null;
			                           }
		                           } ) );

	}
}

